/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.call.dialogs

import android.app.AlertDialog
import android.content.DialogInterface
import android.content.res.Configuration
import android.os.Build
import android.os.Bundle
import androidx.fragment.app.FragmentActivity
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.animation.OvershootInterpolator
import android.widget.Toast
import com.ablanco.zoomy.ZoomListener
import com.ablanco.zoomy.Zoomy
import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.bottom_sheet.BandyerBottomSheetDialog
import com.bandyer.sdk_design.call.layout.BandyerSnapshotDialogLayout
import com.bandyer.sdk_design.dialogs.BandyerDialog
import com.bandyer.sdk_design.extensions.*
import kotlinx.android.synthetic.main.bandyer_snapshot_dialog_layout.*

/**
 *
 * @author kristiyan
 */
class BandyerSnapshotDialog : BandyerDialog<BandyerSnapshotDialog.SnapshotDialogFragment> {

    override val id: String = "BandyerSnapshotDialog"

    override var dialog: SnapshotDialogFragment? = null

    override fun show(activity: androidx.fragment.app.FragmentActivity) {
        if (dialog == null) dialog = SnapshotDialogFragment()
        dialog!!.show(activity.supportFragmentManager, id)
    }

    /**
     * @suppress
     */
    class SnapshotDialogFragment : BandyerBottomSheetDialog() {

        private var confirmDialog: AlertDialog? = null

        private var saved = false

        override val style: Int = R.style.BandyerSDK_DialogStyle_SnapshotPreview

        private var zoomy: Zoomy.Builder? = null

        private var rootView: View? = null

        override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View? {
            rootView = BandyerSnapshotDialogLayout(context!!)
            return rootView
        }

        override fun onStart() {
            super.onStart()
            dialog ?: return
            if (dialog!!.window != null && Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                val window = dialog!!.window
                window?.findViewById<View>(R.id.container)?.fitsSystemWindows = false
                // dark navigation bar icons
                val decorView = window?.decorView
                decorView?.systemUiVisibility = decorView?.systemUiVisibility!! and View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR.inv() and View.SYSTEM_UI_FLAG_LIGHT_NAVIGATION_BAR.inv()
            }
        }

        override fun onConfigurationChanged(newConfig: Configuration) {
            super.onConfigurationChanged(newConfig)
            dismiss()
            BandyerSnapshotDialog().apply {
                context?.scanForFragmentActivity()?.let {
                    this.show(it)
                }
            }
        }

        override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
            super.onViewCreated(view, savedInstanceState)

            // see https://github.com/imablanco/Zoomy/issues/19
            dialog?.window?.decorView?.setOnTouchListener { v, event ->
                bandyer_snapshot_preview.dispatchTouchEvent(event)
                v.performClick()
            }

            zoomy = Zoomy.Builder(this.dialog).target(bandyer_snapshot_preview)
                    .enableImmersiveMode(false)
                    .interpolator(OvershootInterpolator())
                    .animateZooming(true)
                    .doubleTapListener {
                        bandyer_snapshot_share_save_button.visibility = View.VISIBLE
                        bandyer_snapshot_share_close_button.visibility = View.VISIBLE
                        bandyer_snapshot_share_whiteboard_button.visibility = View.VISIBLE
                    }
                    .tapListener {
                        bandyer_snapshot_share_save_button.visibility = View.VISIBLE
                        bandyer_snapshot_share_close_button.visibility = View.VISIBLE
                        bandyer_snapshot_share_whiteboard_button.visibility = View.VISIBLE
                    }
                    .zoomListener(object : ZoomListener {
                        override fun onViewEndedZooming(view: View?) {
                            bandyer_snapshot_share_save_button.visibility = View.VISIBLE
                            bandyer_snapshot_share_close_button.visibility = View.VISIBLE
                            bandyer_snapshot_share_whiteboard_button.visibility = View.VISIBLE
                        }

                        override fun onViewStartedZooming(view: View?) {
                            bandyer_snapshot_share_save_button.visibility = View.INVISIBLE
                            bandyer_snapshot_share_close_button.visibility = View.INVISIBLE
                            bandyer_snapshot_share_whiteboard_button.visibility = View.INVISIBLE
                        }
                    })
            zoomy!!.register()

            bandyer_snapshot_share_save_button.setOnClickListener {
                saved = true
                Toast.makeText(context, context?.resources?.getString(R.string.bandyer_snapshot_saved_in_gallery), Toast.LENGTH_SHORT).show()
            }


            bandyer_snapshot_share_close_button.setOnClickListener {
                if (saved) {
                    dismiss()
                    return@setOnClickListener
                }
                showDialog(context?.resources?.getString(R.string.bandyer_alert_discard_snapshot_title) ?: "",
                        context?.resources?.getString(R.string.bandyer_alert_discard_snapshot_message) ?: "", {
                    dismiss()
                }, {

                })

            }

            bandyer_snapshot_share_whiteboard_button.setOnClickListener {
                if (!saved)
                    showDialog(context?.resources?.getString(R.string.bandyer_alert_save_snapshot_title) ?: "",
                            context?.resources?.getString(R.string.bandyer_alert_save_snapshot_message) ?: "", {

                    }, {

                    })
            }
        }

        override fun onExpanded() {}

        override fun onCollapsed() {}

        override fun onDialogWillShow() {
            behavior?.disableDragging = true
        }

        override fun onSlide(offset: Float) {}

        override fun onStateChanged(newState: Int) {}

        override fun onDismiss(dialog: DialogInterface) {
            super.onDismiss(dialog)
            confirmDialog?.dismiss()
            Zoomy.unregister(bandyer_snapshot_preview)
        }

        private fun showDialog(title: String, body: String, positiveCallback: () -> Unit, negativeCallback: () -> Unit) {
            confirmDialog = AlertDialog.Builder(context)
                    .setTitle(title)
                    .setCancelable(true)
                    .setPositiveButton(context?.resources?.getString(R.string.bandyer_confirm_message)) { dialog, which ->
                        positiveCallback.invoke()
                    }
                    .setNegativeButton(context?.resources?.getString(R.string.bandyer_cancel_message)) { dialog, which ->
                        negativeCallback.invoke()
                    }
                    .setMessage(body)
                    .show()
        }
    }
}