/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.call.bottom_sheet.items

import android.annotation.SuppressLint
import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.bottom_sheet.items.ActionItem
import com.bandyer.sdk_design.buttons.action.toggleable.BandyerActionButtonToggleable
import com.bandyer.sdk_design.call.buttons.action.BandyerActionButtonAudioRoute
import com.bandyer.sdk_design.call.buttons.action.BandyerActionButtonFileShare
import com.bandyer.sdk_design.extensions.setAllEnabled

/**
 * Class representing a Call Action
 * @constructor
 */
sealed class CallAction(viewLayoutRes: Int = 0) : ActionItem(viewLayoutRes) {

    /**
     * Instance of Call Actions
     */
    companion object Items {

        /**
         * Gives a list of all actions used for incoming calls
         * @return List<CallAction> composed of BIG_HANGUP and BIG_ANSWER
         */
        fun getIncomingCallActions() = listOf(BIG_HANGUP(), BIG_ANSWER())

        /**
         * Get all actions for an audio call that can be collapsed
         * @param cameraToggled True if by default the camera should be toggled, false otherwise
         * @param micToggled True if by default the microphone should be toggled, false otherwise
         * @param withChat True if by default the chat should be shown, false otherwise
         * @param withWhiteboard  True if by default the whiteboard should be shown, false otherwise
         * @param withFileShare  True if by default the file share should be shown, false otherwise
         * @param withScreenShare  True if by default the screen share should be shown, false otherwise
         * @return List<CallAction>
         */
        fun getAudioCollapsedActions(cameraToggled: Boolean, micToggled: Boolean, withChat: Boolean, withWhiteboard: Boolean, withFileShare: Boolean, withScreenShare: Boolean): List<CallAction> =
                mutableListOf(
                        MICROPHONE(micToggled),
                        CAMERA(cameraToggled),
                        SWITCH_CAMERA()
                ).also {
                    if (withChat) it.add(CHAT())
                    if (withWhiteboard) it.add(WHITEBOARD())
                    it.add(AUDIOROUTE())
                    if (withFileShare) it.add(FILE_SHARE())
                    if (withScreenShare) it.add(SCREEN_SHARE())
                    if (it.size < 3) it.add(1, HANGUP())
                    else it.add(3, HANGUP())
                }

        /**
         * Get all actions for an audio not upgradable call
         * @param micToggled True if by default the microphone should be toggled, false otherwise
         * @param withChat True if by default the chat should be shown, false otherwise
         * @param withWhiteboard  True if by default the whiteboard should be shown, false otherwise
         * @param withFileShare  True if by default the file share should be shown, false otherwise
         * @param withScreenShare  True if by default the screen share should be shown, false otherwise
         * @return List<CallAction>
         */
        fun getAudioNotUpgradableActions(micToggled: Boolean, withChat: Boolean, withWhiteboard: Boolean, withFileShare: Boolean, withScreenShare: Boolean): List<CallAction> =
                mutableListOf<CallAction>(
                        MICROPHONE(micToggled)
                ).also {
                    if (withChat) it.add(CHAT())
                    if (withWhiteboard) it.add(WHITEBOARD())
                    it.add(AUDIOROUTE())
                    if (withFileShare) it.add(FILE_SHARE())
                    if (withScreenShare) it.add(SCREEN_SHARE())
                    if (it.size < 3) it.add(1, HANGUP())
                    else it.add(3, HANGUP())
                }

        /**
         * Get all actions for an audio&Video call that can be collapsed
         * @param cameraToggled True if by default the camera should be toggled, false otherwise
         * @param micToggled True if by default the microphone should be toggled, false otherwise
         * @param withChat True if by default the chat should be shown, false otherwise
         * @param withWhiteboard  True if by default the whiteboard should be shown, false otherwise
         * @param withFileShare  True if by default the file share should be shown, false otherwise
         * @param withScreenShare  True if by default the screen share should be shown, false otherwise
         * @return List<CallAction>
         */
        fun getAudioVideoCallCollapsedActions(cameraToggled: Boolean, micToggled: Boolean, withChat: Boolean, withWhiteboard: Boolean, withFileShare: Boolean, withScreenShare: Boolean): List<CallAction> =
                mutableListOf(
                        MICROPHONE(micToggled),
                        CAMERA(cameraToggled),
                        SWITCH_CAMERA()
                ).also {
                    if (withChat) it.add(CHAT())
                    if (withWhiteboard) it.add(WHITEBOARD())
                    it.add(AUDIOROUTE())
                    if (withFileShare) it.add(FILE_SHARE())
                    if (withScreenShare) it.add(SCREEN_SHARE())
                    if (it.size < 3) it.add(1, HANGUP())
                    else it.add(3, HANGUP())
                }
    }

    /**
     * Togglable Call Action
     * @property toggled true to select, false otherwise
     * @constructor
     */
    abstract class TogglableCallAction(mToggled: Boolean, viewLayoutRes: Int = 0) : CallAction(viewLayoutRes) {

        /**
         * @suppress
         */
        protected abstract val buttonId: Int

        init {
            toggle(mToggled)
        }

        /**
         * If action button is toggled
         */
        var toggled = false
            private set

        /**
         * Toggle the action button
         */
        fun toggle() {
            val button = itemView?.findViewById<BandyerActionButtonToggleable>(buttonId)
            button?.toggle()
            toggled = button?.toggled == true
        }

        /**
         * Toggle the action button
         * @param enable true to enable, false otherwise
         */
        fun toggle(enable: Boolean) {
            itemView?.findViewById<BandyerActionButtonToggleable>(buttonId)?.toggle(enable)
            toggled = enable
        }

        override fun onReady() {
            toggle(toggled)
        }
    }

    /**
     * Camera call action item
     * @property toggled true or false to toggle
     * @constructor
     */
    open class CAMERA(mToggled: Boolean, viewLayoutRes: Int = R.layout.bandyer_call_action_item_camera) : TogglableCallAction(mToggled, viewLayoutRes) {

        override val buttonId: Int
            get() = R.id.video
    }

    /**
     * Microphone call action item
     * @property toggled true or false to toggle
     * @constructor
     */
    open class MICROPHONE(mToggled: Boolean, viewLayoutRes: Int = R.layout.bandyer_call_action_item_microphone) : TogglableCallAction(mToggled, viewLayoutRes) {

        override val buttonId: Int
            get() = R.id.microphone
    }

    /**
     * Options call action item
     * @property switchWith CallAction to replace the options with
     * @constructor
     */
    open class OPTIONS(val switchWith: CallAction, viewLayoutRes: Int = R.layout.bandyer_call_action_item_options) : CallAction(viewLayoutRes)

    /**
     * Switch camera call action item
     * @constructor
     */
    open class SWITCH_CAMERA(viewLayoutRes: Int = R.layout.bandyer_call_action_item_switch_camera) : CallAction(viewLayoutRes)

    /**
     * Participants call action item
     * @constructor
     */
    open class PARTICIPANTS(viewLayoutRes: Int = R.layout.bandyer_call_action_item_participants) : CallAction(viewLayoutRes)

    /**
     * Open Chat call action item
     * @constructor
     */
    open class CHAT(viewLayoutRes: Int = R.layout.bandyer_call_action_item_chat) : CallAction(viewLayoutRes)

    /**
     * AudioRoute call action item
     * @property mCurrent AudioRoute? current AudioRoute device
     * @constructor
     */
    open class AUDIOROUTE(viewLayoutRes: Int = R.layout.bandyer_call_action_item_audioroute) : CallAction(viewLayoutRes) {

        var mCurrent: AudioRoute? = null

        /**
         * Change current AudioRoute device
         * @param item new AudioRoute?
         */
        fun setCurrent(item: AudioRoute?) {
            if (item == null)
                return

            val view = itemView?.findViewById<BandyerActionButtonAudioRoute>(R.id.audio_route)
            view?.post {
                when (item) {
                    is AudioRoute.BLUETOOTH -> view.setBluetoothIcon()
                    is AudioRoute.WIRED_HEADSET -> view.setWiredHeadsetIcon()
                    is AudioRoute.LOUDSPEAKER -> view.setLoudSpeakerIcon()
                    is AudioRoute.EARPIECE -> view.setEarpieceIcon()
                    is AudioRoute.MUTED -> view.setMutedIcon()
                }
            }
            mCurrent = item
        }

        override fun onReady() {
            setCurrent(mCurrent)
        }
    }

    /**
     * Hangup call action item
     * @constructor
     */
    open class BIG_HANGUP(viewLayoutRes: Int = R.layout.bandyer_call_action_item_big_hangup) : CallAction(viewLayoutRes)

    /**

     * Hangup call action item
     * @constructor
     */
    open class HANGUP(viewLayoutRes: Int = R.layout.bandyer_call_action_item_hangup) : CallAction(viewLayoutRes)

    /**
     * Answer call action item
     * @constructor
     */
    open class BIG_ANSWER(viewLayoutRes: Int = R.layout.bandyer_call_action_item_big_answer) : CallAction(viewLayoutRes)


    /**
     * Whiteboard call action item
     * @constructor
     */
    open class WHITEBOARD(viewLayoutRes: Int = R.layout.bandyer_call_action_item_whiteboard) : CallAction(viewLayoutRes)


    /**
     * Upload call action item
     * @constructor
     */
    open class FILE_SHARE(viewLayoutRes: Int = R.layout.bandyer_call_action_item_file_share) : CallAction(viewLayoutRes) {

        private var oldLabelText: String? = null

        /**
         * If action button is enabled
         */
        var enabled = true
            private set

        override fun onReady() {
            val action = itemView!!.findViewById<BandyerActionButtonFileShare>(R.id.action_file_share)
            if (oldLabelText == null) oldLabelText = action.label?.text?.toString()
            action.label?.text = oldLabelText
            action.setAllEnabled(enabled)
        }

        /**
         * Set the Upload progress
         * @param progress percentage from 0 to 100
         */
        @SuppressLint("SetTextI18n")
        fun setUploadProgress(progress: Float) {
            itemView ?: return
            val action = itemView!!.findViewById<BandyerActionButtonFileShare>(R.id.action_file_share)
            if (progress < 100) {
                action.label?.text = "${progress.toInt()}%"
            } else {
                action.label?.text = oldLabelText
            }
        }

        /**
         * Remove the progress
         */
        fun removeUploadProgress() {
            itemView ?: return
            val action = itemView!!.findViewById<BandyerActionButtonFileShare>(R.id.action_file_share)
            action.label?.text = oldLabelText
        }

        /**
         * Enable the action
         */
        fun enable() {
            itemView ?: return
            enabled = true
            val action = itemView!!.findViewById<BandyerActionButtonFileShare>(R.id.action_file_share)
            action.setAllEnabled(true)
        }

        /**
         * Disable the action
         */
        fun disable() {
            itemView ?: return
            enabled = false
            val action = itemView!!.findViewById<BandyerActionButtonFileShare>(R.id.action_file_share)
            action.setAllEnabled(false)
        }
    }

    /**
     * Microphone call action item
     * @property toggled true or false to toggle
     * @constructor
     */
    open class SCREEN_SHARE(mToggled: Boolean = false, viewLayoutRes: Int = R.layout.bandyer_call_action_item_screenshare) : TogglableCallAction(mToggled, viewLayoutRes) {

        override val buttonId: Int
            get() = R.id.screen_share
    }

    /**
     * Whiteboard call action item
     * @constructor
     */
    open class EMPTY(viewLayoutRes: Int = R.layout.bandyer_call_action_item_empty) : CallAction(viewLayoutRes)

    /**
     * Called when the layout has been inflated
     */
    override fun onReady() {}
}