/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.call.bottom_sheet.items

import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.bottom_sheet.items.ActionItem
import com.bandyer.sdk_design.textviews.BandyerTextView

/**
 * Class representing an AudioRoute
 * @property name name of the AudioRoute
 * @constructor
 */
sealed class AudioRoute(val name: String? = null, viewLayoutRes: Int = 0) : ActionItem(viewLayoutRes) {

    /**
     * Instance of AudioRoute
     */
    companion object {

        /**
         * All audioRoute items
         */
        val values = listOf(LOUDSPEAKER(), BLUETOOTH(), EARPIECE(), WIRED_HEADSET(), MUTED())

        /**
         * Get an AudioRoute based on the device type and name
         * @param deviceType type of device
         * @param deviceName name of device
         * @return AudioRoute
         */
        fun getAudioRoute(deviceType: AudioRoute?, deviceName: String?): AudioRoute {
            return when (deviceType) {
                is LOUDSPEAKER -> LOUDSPEAKER(deviceName)
                is BLUETOOTH -> BLUETOOTH(deviceName)
                is EARPIECE -> EARPIECE(deviceName)
                is WIRED_HEADSET -> WIRED_HEADSET(deviceName)
                is MUTED -> MUTED(deviceName)
                else -> LOUDSPEAKER(deviceName)
            }
        }
    }

    /**
     * Loudspeaker AudioRoute Item
     * @constructor
     */
    open class LOUDSPEAKER(name: String? = null) : AudioRoute(name, R.layout.bandyer_audioroute_item_loudspeaker)

    /**
     * Bluetooth AudioRoute Item
     * @constructor
     */
    open class BLUETOOTH(name: String? = null) : AudioRoute(name, R.layout.bandyer_audioroute_item_bluetooth)

    /**
     * Earpiece AudioRoute Item
     * @constructor
     */
    open class EARPIECE(name: String? = null) : AudioRoute(name, R.layout.bandyer_audioroute_item_earpiece)

    /**
     * Wired headset AudioRoute Item
     * @constructor
     */
    open class WIRED_HEADSET(name: String? = null) : AudioRoute(name, R.layout.bandyer_audioroute_item_headset)

    /**
     * Muted AudioRoute Item
     * @constructor
     */
    open class MUTED(name: String? = null) : AudioRoute(name, R.layout.bandyer_audioroute_item_muted)

    /**
     * Called when the layout has been inflated
     */
    override fun onReady() {
        val deviceName = name ?: return
        val item = itemView?.findViewById<BandyerTextView>(R.id.audio_item)
        item?.text = deviceName
    }
}