/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.call.bottom_sheet

import android.animation.ValueAnimator
import android.os.Bundle
import android.view.View
import androidx.appcompat.app.AppCompatActivity
import androidx.fragment.app.FragmentActivity
import androidx.recyclerview.widget.RecyclerView
import com.bandyer.sdk_design.bottom_sheet.BandyerBottomSheet
import com.bandyer.sdk_design.bottom_sheet.BandyerClickableBottomSheet
import com.bandyer.sdk_design.bottom_sheet.behaviours.BandyerBottomSheetBehaviour
import com.bandyer.sdk_design.bottom_sheet.behaviours.BandyerBottomSheetBehaviour.Companion.STATE_ANCHOR_POINT
import com.bandyer.sdk_design.bottom_sheet.behaviours.BandyerBottomSheetBehaviour.Companion.STATE_COLLAPSED
import com.bandyer.sdk_design.bottom_sheet.behaviours.BandyerBottomSheetBehaviour.Companion.STATE_EXPANDED
import com.bandyer.sdk_design.bottom_sheet.items.ActionItem
import com.bandyer.sdk_design.bottom_sheet.view.BottomSheetLayoutType
import com.bandyer.sdk_design.call.bottom_sheet.items.AudioRoute
import com.bandyer.sdk_design.call.bottom_sheet.items.CallAction
import com.bandyer.sdk_design.extensions.dp2px
import com.bandyer.sdk_design.utils.systemviews.SystemViewLayoutOffsetListener
import com.bandyer.sdk_design.utils.systemviews.implementation.SystemViewControlsAware
import com.bandyer.sdk_design.utils.systemviews.implementation.SystemViewControlsAwareInstance
import kotlinx.android.synthetic.main.bandyer_bottom_sheet_widget_layout.view.*


/**
 * Call BottomSheet to display actions and interact with the call
 * @param callActionItems list of actions to display
 * @constructor
 * @author kristiyan
 */
@Suppress("UNCHECKED_CAST")
open class CallBottomSheet<T>(context: AppCompatActivity,
                              private val callActionItems: List<CallAction>,
                              bottomSheetStyle: Int) : BandyerClickableBottomSheet<T>(
        context,
        callActionItems as List<T>,
        callActionItems.size.takeIf { it < MAX_ITEMS_PER_ROW } ?: MAX_ITEMS_PER_ROW,
        0,
        BottomSheetLayoutType.GRID,
        bottomSheetStyle) where T : ActionItem {

    private var camera: CallAction.CAMERA? = null
    private var mic: CallAction.MICROPHONE? = null
    private var audioRoute: CallAction.AUDIOROUTE? = null
    private var currentAudioRoute: AudioRoute? = null

    private var animationStartOffset = -1f
    private var animationEndState = -1
    private var lineAnimator: ValueAnimator? = null

    /**
     * Describes if CallBottomSheet can be collapsed
     */
    var collapsible = true

    private var cameraToggled = false
    private var micToggled = false

    /**
     * Singleton of the call bottom sheet
     */
    companion object {
        /**
         * Max visible items per row
         */
        const val MAX_ITEMS_PER_ROW = 4

        /**
         * Default anchor offset
         */
        const val ANCHOR_OFFSET = 158f
    }

    init {
        firstOrNull(CallAction.OPTIONS::class.java).let {
            camera = firstOrNull(CallAction.CAMERA::class.java)
                    ?: it?.switchWith as? CallAction.CAMERA
            mic = firstOrNull(CallAction.MICROPHONE::class.java)
                    ?: it?.switchWith as? CallAction.MICROPHONE
        }
        recyclerView?.overScrollMode = RecyclerView.OVER_SCROLL_NEVER
    }

    override fun saveInstanceState(saveInstanceState: Bundle?): Bundle? {
        saveInstanceState?.putString("audioRouteType", audioRoute?.mCurrent?.toString())
        saveInstanceState?.putString("audioRouteDeviceName", audioRoute?.mCurrent?.name)
        saveInstanceState?.putBoolean("cameraIsToggled", camera?.toggled == true)
        saveInstanceState?.putBoolean("micIsToggled", mic?.toggled == true)
        return onSaveInstanceState(saveInstanceState, "action")
    }

    override fun restoreInstanceState(bundle: Bundle?) {
        onRestoreInstanceState(bundle, "action")

        val audioRouteType = bundle?.getString("audioRouteType")
        val audioRouteDeviceName = bundle?.getString("audioRouteDeviceName")
        if (!audioRouteType.isNullOrBlank())
            updateAudioRouteIcon(AudioRoute.getAudioRoute(AudioRoute.values.firstOrNull { it.toString() == audioRouteType }, audioRouteDeviceName))

        bundle?.getBoolean("micIsToggled", false)?.let {
            mic?.toggle(it)
        }

        bundle?.getBoolean("cameraIsToggled", false)?.let {
            camera?.toggle(it)
        }
    }


    /**
     * Collapse the bottomSheet
     */
    fun collapse() {
        collapse(lineView, mContext.get()?.dp2px(15f))
    }

    /**
     * Show the bottomSheet
     * @param collapsible true if it should be collapsible, false otherwise
     * @param fixed true if it should not move, false otherwise
     * @param collapsed optional initial collapsed state
     */
    fun show(collapsible: Boolean, fixed: Boolean? = false, collapsed: Boolean = false) {
        super.show()
        animationStartOffset = -1f
        animationEndState = -1
        animationEnabled = fixed == false

        this.collapsible = collapsible

        if (callActionItems.size <= MAX_ITEMS_PER_ROW) {
            bottomSheetBehavior!!.disableDragging = true
        }

        if (fixed == true) {
            bottomSheetBehavior!!.isHideable = false
            bottomSheetBehavior!!.skipCollapsed = true
            bottomSheetBehavior!!.disableDragging = true
            bottomSheetBehavior!!.skipCollapsed = true
            expand()
            return
        }

        var peekHeight = 0
        val anchorOffset: Int

        when {
            collapsible -> {
                bottomSheetLayoutContent.post {
                    peekHeight = bottomSheetLayoutContent.bottom_sheet_root.top + lineView?.height!! + (mContext.get()?.dp2px(15f)
                            ?: 0)
                }
                anchorOffset = mContext.get()?.dp2px(ANCHOR_OFFSET) ?: 0
                animationEndState = if (collapsed && collapsible) STATE_COLLAPSED else STATE_ANCHOR_POINT
                bottomSheetBehavior!!.skipCollapsed = false
            }
            else -> {
                peekHeight = mContext.get()?.dp2px(ANCHOR_OFFSET) ?: 0
                anchorOffset = mContext.get()?.dp2px(ANCHOR_OFFSET) ?: 0
                bottomSheetBehavior!!.skipCollapsed = true
            }
        }

        bottomSheetLayoutContent.post {
            bottomSheetBehavior ?: return@post

            with(bottomSheetBehavior!!) {
                this.peekHeight = peekHeight
                this.anchorOffset = anchorOffset
                skipAnchor = false
                bottomSheetBehavior!!.isHideable = false
                state =  if (collapsed && collapsible) STATE_COLLAPSED else STATE_ANCHOR_POINT
            }

            if (animationEndState == -1) return@post
            animationStartOffset = bottomSheetBehavior!!.getStableStateSlideOffset(animationEndState)
        }

        bottomSheetLayoutContent.toggleLineViewStyle(bottomSheetBehavior?.skipCollapsed == true)

        if (collapsed && collapsible) bottomSheetLayoutContent.backgroundView.alpha = 0f

        lineView?.visibility =
                if (recyclerView?.adapter?.itemCount ?: 0 <= MAX_ITEMS_PER_ROW && bottomSheetBehavior!!.skipCollapsed)
                    View.INVISIBLE
                else View.VISIBLE

        lineView?.setOnClickListener {
            if (state == STATE_COLLAPSED)
                anchor()
            else if (bottomSheetBehavior!!.skipCollapsed)
                expand()
        }

        camera?.toggle(cameraToggled)
        mic?.toggle(micToggled)
    }

    override fun slideAnimationReady(bottomSheet: BandyerBottomSheet?, state: Int, slideOffset: Float) {
        if (!animationEnabled) return
        if (state == animationEndState) animationStartOffset = slideOffset
    }

    override fun slideAnimationUpdate(bottomSheet: BandyerBottomSheet?, slideOffset: Float) {
        super.slideAnimationUpdate(bottomSheet, slideOffset)
        if (!animationEnabled || bottomSheetBehavior?.lastStableState == state) return
        val isCollapsed = bottomSheetBehavior?.skipCollapsed == true || slideOffset <= 0f
        bottomSheetLayoutContent.toggleLineViewStyle(isCollapsed)
    }

    override fun onSettling() {
        super.onSettling()
        bottomSheetLayoutContent.backgroundView.alpha = 1f
    }

    override fun onExpanded() {
        super.onExpanded()
        replaceOptionItem()
    }

    override fun onDragging() {
        super.onDragging()
        bottomSheetLayoutContent.backgroundView.alpha = 1f
    }

    override fun onHidden() {
        super.onHidden()
        lineAnimator?.removeAllUpdateListeners()
        replaceOptionItem(true)
        cameraToggled = camera?.toggled == true
        micToggled = mic?.toggled == true
    }

    override fun onCollapsed() {
        super.onCollapsed()
        replaceOptionItem(true)
        bottomSheetLayoutContent.toggleLineViewStyle(true)
        bottomSheetLayoutContent.backgroundView.alpha = 0f
    }

    override fun onAnchor() {
        super.onAnchor()
        replaceOptionItem(true)
        bottomSheetLayoutContent.toggleLineViewStyle(bottomSheetBehavior?.skipCollapsed == true)
    }

    /**
     * Update the currentAudio route displayed in the actions
     * @param audioRoute new AudioRoute? to be displayed
     */
    fun updateAudioRouteIcon(audioRoute: AudioRoute?) {
        this.audioRoute = firstOrNull(CallAction.AUDIOROUTE::class.java)
        this.audioRoute?.setCurrent(audioRoute)
        currentAudioRoute = audioRoute
        if (state == STATE_EXPANDED) replaceOptionItem()
    }

    private fun replaceOptionItem(withOptionActionItem: Boolean = false) {
        val options = callActionItems.firstOrNull { it is CallAction.OPTIONS } as? CallAction.OPTIONS
                ?: return

        val oldItem = firstOrNull(options.switchWith::class.java) ?: options

        if (withOptionActionItem && oldItem is CallAction.OPTIONS) return

        val newItem = if (!withOptionActionItem) options.switchWith else options

        (newItem as? CallAction.AUDIOROUTE)?.setCurrent(currentAudioRoute)

        replaceItems(oldItem, newItem)
    }
}