/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.call.bottom_sheet

import android.os.Bundle
import androidx.appcompat.app.AppCompatActivity
import com.bandyer.sdk_design.bottom_sheet.BandyerBottomSheet
import com.bandyer.sdk_design.bottom_sheet.BandyerSelectableBottomSheet
import com.bandyer.sdk_design.bottom_sheet.items.ActionItem
import com.bandyer.sdk_design.bottom_sheet.view.BottomSheetLayoutType
import com.bandyer.sdk_design.call.bottom_sheet.items.AudioRoute

/**
 * AudioRoute BottomSheet to display the available audioRoutes of the device
 * @param onAudioRoutesRequest used to request available audioRoutes
 * @constructor
 * @author kristiyan
 */
@Suppress("UNCHECKED_CAST")
class AudioRouteBottomSheet<T : ActionItem>(context: AppCompatActivity,
                                            audioRouteItems: List<AudioRoute>?,
                                            initial_selection: Int = -1,
                                            bottomSheetStyle: Int,
                                            var onAudioRoutesRequest: OnAudioRouteBottomSheetListener?) : BandyerSelectableBottomSheet<T>(
        context,
        initial_selection,
        audioRouteItems as List<T>? ?: listOf<T>(),
        0, 0,
        BottomSheetLayoutType.LIST,
        bottomSheetStyle) {

    /**
     * Returns the current selected Audio Route
     */
    var mCurrentAudioRoute: AudioRoute? = null

    init {
        animationEnabled = true
    }

    override fun show() {
        super.show()
        onAudioRoutesRequest?.onAudioRoutesRequested()?.let { setItems(it) }
        selectItem(mCurrentAudioRoute)
        bottomSheetBehavior!!.skipCollapsed = true
        bottomSheetBehavior!!.isHideable = true
        bottomSheetBehavior!!.skipAnchor = true
        bottomSheetLayoutContent.backgroundView.alpha = 1f
        expand()
    }

    override fun saveInstanceState(saveInstanceState: Bundle?): Bundle? {
        saveInstanceState?.putString("currentAudioRoute", mCurrentAudioRoute!!.javaClass.simpleName)
        return onSaveInstanceState(saveInstanceState, "audio")
    }

    override fun restoreInstanceState(bundle: Bundle?) {
        onRestoreInstanceState(bundle, "audio")
        val currentAudioRoute = bundle?.getString("currentAudioRoute")
        AudioRoute.values.firstOrNull { it.javaClass.simpleName == currentAudioRoute }?.let {
            mCurrentAudioRoute = it
        }
    }

    /**
     * Select another audioRoute
     * @param audioRoute AudioRoute? to select
     */
    fun selectAudioRoute(audioRoute: AudioRoute?) {
        if (audioRoute == null || mCurrentAudioRoute == AudioRoute.MUTED() || mCurrentAudioRoute == audioRoute)
            return

        mCurrentAudioRoute = audioRoute
        selectItem(audioRoute)
    }

    /**
     * Remove an audioRoute from the list
     * @param audioRoute AudioRoute to remove
     */
    fun removeAudioRouteItem(audioRoute: AudioRoute) {
        removeItem(audioRoute)
    }

    /**
     * Add a new audioRoute
     * @param audioRoute AudioRoute to add
     */
    fun addAudioRouteItem(audioRoute: AudioRoute) {
        val position = if (fastAdapter.itemCount == 0) 0
        else onAudioRoutesRequest?.onAudioRoutesRequested()?.indexOf(audioRoute)?.takeIf { it >= 0 }
                ?: 0
        addItem(audioRoute, position)
    }
}

/**
 * Listener called when the available audioRoute are request to be drawn
 */
interface OnAudioRouteBottomSheetListener {

    /**
     * Return the list  of available audioRoutes
     * @return List<AudioRoute>?
     */
    fun onAudioRoutesRequested(): List<AudioRoute>?
}