/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.buttons.action.toggleable

import android.content.Context
import androidx.annotation.AnyRes
import android.util.AttributeSet
import com.airbnb.paris.annotations.Attr
import com.airbnb.paris.annotations.Styleable
import com.bandyer.sdk_design.R2
import com.bandyer.sdk_design.buttons.action.BandyerActionButton
import com.bandyer.sdk_design.extensions.StyleCompat

/**
 * This base class represents a toggleable borderless button.
 * Enabled and disabled states can be styled overriding a custom borderless button style
 * with custom style attributes 'BandyerSDKDesign.Button.Togglable.toggleOnSrc' and 'BandyerSDKDesign.Button.Togglable.toggleOffSrc'.
 */
@Styleable("BandyerSDKDesign_ActionButton_Toggleable")
open class BandyerActionButtonToggleable @JvmOverloads constructor(context: Context, attrs: AttributeSet? = null, defStyleAttr: Int? = null) : BandyerActionButton(context, attrs, defStyleAttr) {

    private var buttonToggleOnStyle: Int = 0
    private var buttonToggleOffStyle: Int = 0

    private var labelToggleOnStyle: Int = 0
    private var labelToggleOffStyle: Int = 0

    /**
     * Toggles drawable state, changing the drawable
     * 'BandyerSDKDesign.ActionButton.Toggleable.toggleOnSrc' and 'BandyerSDKDesign.ActionButton.Toggleable.toggleOffSrc'
     * custom style attributes with input state
     */
    var toggled: Boolean = false
        set(value) {
            if (value) {
                StyleCompat.style(button).apply(buttonToggleOnStyle)
                StyleCompat.style(label).apply(labelToggleOnStyle)
            } else {
                StyleCompat.style(button).apply(buttonToggleOffStyle)
                StyleCompat.style(label).apply(labelToggleOffStyle)
            }
            field = value
        }


    init {
        StyleCompat.style(this).apply(attrs)
    }

    /**
     * Toggles drawable state, changing the drawable
     * 'BandyerSDKDesign.ActionButton.Toggleable.toggleOnSrc' and 'BandyerSDKDesign.ActionButton.Toggleable.toggleOffSrc'
     * custom style attributes with input state
     */
    fun toggle() {
        toggled = !toggled
    }

    /**
     * Set the button style of toggleON status
     * @param buttonToggleOnStyle the status to be set
     */
    @Attr(R2.styleable.BandyerSDKDesign_ActionButton_Toggleable_bandyer_toggleOnButtonStyle)
    fun buttonToggleOn(@AnyRes buttonToggleOnStyle: Int = 0) {
        this.buttonToggleOnStyle = buttonToggleOnStyle
    }

    /**
     * Set the button style of toggleOFF status
     * @param buttonToggleOffStyle the status to be set
     */
    @Attr(R2.styleable.BandyerSDKDesign_ActionButton_Toggleable_bandyer_toggleOffButtonStyle)
    fun buttonToggleOff(@AnyRes buttonToggleOffStyle: Int = 0) {
        this.buttonToggleOffStyle = buttonToggleOffStyle
    }

    /**
     *  Set the label style of toggleON status
     * @param labelToggleOnStyle the status to be set
     */
    @Attr(R2.styleable.BandyerSDKDesign_ActionButton_Toggleable_bandyer_toggleOnLabelStyle)
    fun labelToggleOn(@AnyRes labelToggleOnStyle: Int = 0) {
        this.labelToggleOnStyle = labelToggleOnStyle
    }

    /**
     *  Set the label style of toggleOFF status
     * @param labelToggleOffStyle the status to be set
     */
    @Attr(R2.styleable.BandyerSDKDesign_ActionButton_Toggleable_bandyer_toggleOffLabelStyle)
    fun labelToggleOff(@AnyRes labelToggleOffStyle: Int = 0) {
        this.labelToggleOffStyle = labelToggleOffStyle
    }

    /**
     * Automatically toggles drawable reading
     * 'BandyerSDKDesign.Button.Togglable.toggleOnSrc' and 'BandyerSDKDesign.Button.Togglable.toggleOffSrc'
     * custom style attributes
     */
    @Attr(R2.styleable.BandyerSDKDesign_ActionButton_Toggleable_bandyer_toggled)
    fun toggle(enabled: Boolean) {
        toggled = enabled
    }

}