/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.buttons


import android.annotation.SuppressLint
import android.content.Context
import android.graphics.PorterDuff
import android.os.Build
import androidx.annotation.ColorInt
import androidx.core.graphics.drawable.DrawableCompat
import androidx.appcompat.widget.AppCompatImageButton
import android.util.AttributeSet
import com.airbnb.paris.annotations.Attr
import com.airbnb.paris.annotations.Styleable
import com.bandyer.sdk_design.Paris
import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.R2
import com.bandyer.sdk_design.buttons.action.BandyerActionButton


/**
 * Bandyer Tintable ImageButton
 * @author kristiyan
 */
@Styleable("BandyerSDKDesign_ImageButton")
open class BandyerImageButton @JvmOverloads constructor(context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0) : AppCompatImageButton(context, attrs, defStyleAttr) {

    init {
        Paris.styleBuilder(this).add(R.style.BandyerSDKDesign_ImageButton).add(attrs).apply()
    }

    /**
     * Method to change the color of the image
     * @param color color to apply
     */
    @Attr(R2.styleable.BandyerSDKDesign_ImageButton_bandyer_tint)
    fun tint(@ColorInt color: Int) {
        drawable?.let {
            val drawable = DrawableCompat.wrap(it).mutate()
            setImageDrawable(drawable)
            DrawableCompat.setTint(drawable, color)
        }
    }


    /**
     * Method to change the background color of the image
     * @param color color to apply
     */
    @SuppressLint("RestrictedApi")
    @Attr(R2.styleable.BandyerSDKDesign_ImageButton_bandyer_backgroundTint)
    fun backgroundTint(@ColorInt color: Int) {
        background ?: return
        val drawable = background.mutate()
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP)
            drawable.setColorFilter(color, PorterDuff.Mode.SRC_IN)
        else
            DrawableCompat.setTint(drawable, color)
        background = drawable
    }


    /**
     * Method to change the tint mode of the image
     * @param mode  PorterDuff.Mode to apply
     */
    @Attr(R2.styleable.BandyerSDKDesign_ImageButton_bandyer_tintModeValue)
    fun setTintModeValue(mode: Int) {
        setTintMode(getPorterDuffMode(mode))
    }

    /**
     * Method to change the tint mode of the image
     * @param mode PorterDuff.Mode to apply
     */
    fun setTintMode(mode: PorterDuff.Mode) {
        drawable?.let {
            val drawable = DrawableCompat.wrap(it).mutate()
            setImageDrawable(drawable)
            DrawableCompat.setTintMode(drawable, mode)
        }
    }

    private fun getPorterDuffMode(value: Int): PorterDuff.Mode {
        when (value) {
            0 -> return PorterDuff.Mode.CLEAR
            1 -> return PorterDuff.Mode.SRC
            2 -> return PorterDuff.Mode.DST
            3 -> return PorterDuff.Mode.SRC_OVER
            4 -> return PorterDuff.Mode.DST_OVER
            5 -> return PorterDuff.Mode.SRC_IN
            6 -> return PorterDuff.Mode.DST_IN
            7 -> return PorterDuff.Mode.SRC_OUT
            8 -> return PorterDuff.Mode.DST_OUT
            9 -> return PorterDuff.Mode.SRC_ATOP
            10 -> return PorterDuff.Mode.DST_ATOP
            11 -> return PorterDuff.Mode.XOR
            16 -> return PorterDuff.Mode.DARKEN
            17 -> return PorterDuff.Mode.LIGHTEN
            13 -> return PorterDuff.Mode.MULTIPLY
            14 -> return PorterDuff.Mode.SCREEN
            12 -> return PorterDuff.Mode.ADD
            15 -> return PorterDuff.Mode.OVERLAY
            else -> return PorterDuff.Mode.CLEAR
        }
    }

    /**
     * Returns true if this button is a bottom sheet's child, false otherwise
     * @return Boolean
     */
    fun isBottomSheetButton(): Boolean = this.tag == BandyerActionButton.BOTTOM_SHEET_TAG
}
