/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.bottom_sheet.view

import android.content.Context
import android.content.res.Configuration
import android.os.Build
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.view.View.OnFocusChangeListener
import android.widget.LinearLayout
import androidx.annotation.AnyRes
import androidx.appcompat.widget.AppCompatImageButton
import androidx.recyclerview.widget.RecyclerView
import com.airbnb.paris.annotations.Attr
import com.airbnb.paris.annotations.Styleable
import com.airbnb.paris.annotations.StyleableChild
import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.R2
import com.bandyer.sdk_design.extensions.StyleCompat
import com.bandyer.sdk_design.textviews.BandyerTextViewTitle
import kotlinx.android.synthetic.main.bandyer_bottom_sheet_widget_layout.view.*

/**
 * The bottomSheet Layout
 * @author kristiyan
 */
@Styleable("BandyerSDKDesign_Widget_BottomSheetLayout")
class BottomSheetLayoutContent @kotlin.jvm.JvmOverloads constructor(context: Context, attrs: AttributeSet? = null, defStyleAttr: Int? = null)
    : LinearLayout(context, attrs, defStyleAttr ?: 0) {

    /**
     * Title of the bottomSheet
     */
    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_BottomSheetLayout_bandyer_titleStyle)
    var titleView: BandyerTextViewTitle? = null

    /**
     * Header line of the bottomSheet
     */
    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_BottomSheetLayout_bandyer_lineStyle)
    var lineView: AppCompatImageButton? = null

    /**
     * RecyclerView of the bottomSheet
     */
    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_BottomSheetLayout_bandyer_recyclerViewStyle)
    var recyclerView: RecyclerView? = null

    /**
     * Background of the bottomSheet
     */
    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_BottomSheetLayout_bandyer_backgroundStyle)
    var backgroundView: View = View(context)

    private var layoutChangedListener = object : OnLayoutChangeListener {
        override fun onLayoutChange(v: View?, left: Int, top: Int, right: Int, bottom: Int, oldLeft: Int, oldTop: Int, oldRight: Int, oldBottom: Int) {
            if (backgroundView.layoutParams == null || this@BottomSheetLayoutContent.layoutParams == null) return
            backgroundView.y = oldTop.toFloat()
            backgroundView.layoutParams.height = this@BottomSheetLayoutContent.layoutParams.height
            backgroundView.requestLayout()
        }
    }

    /**
     * Line view style res
     */
    private var lineStyleRes: Int = 0

    /**
     * Collapsed line view style red
     */
    private var collapsedLineStyleRes: Int = 0

    init {
        LayoutInflater.from(context).inflate(R.layout.bandyer_bottom_sheet_widget_layout, this)
        id = R.id.bandyer_id_bottom_sheet_layout_content
        titleView = title!!
        lineView = line!!
        recyclerView = recycler_view!!

        recyclerView!!.onFocusChangeListener = OnFocusChangeListener { v, hasFocus ->
            if (hasFocus)
                recyclerView?.layoutManager?.findViewByPosition(0)?.requestFocus()
        }

        StyleCompat.styleBuilder(this)
                .add(attrs)
                .add(R.style.BandyerSDKDesign_Widget_BottomSheetLayout)
                .apply()
    }

    /**
     * Set visibility of bottom sheet layout and its background view
     */
    override fun setVisibility(visibility: Int) {
        super.setVisibility(visibility)
        backgroundView.visibility = visibility
    }

    /**
     * Called when device configuration changes
     * @param newConfig new device configuration
     */
    override fun onConfigurationChanged(newConfig: Configuration?) {
        super.onConfigurationChanged(newConfig)
        post { updateBackgroundView() }

    }

    /**
     * Update background view position and height
     */
    fun updateBackgroundView() {
        if (backgroundView.layoutParams == null || this@BottomSheetLayoutContent.layoutParams == null) return
        backgroundView.y = y
        backgroundView.layoutParams.height = this@BottomSheetLayoutContent.layoutParams.height
        backgroundView.invalidate()
        backgroundView.requestLayout()
    }

    /**
     * Called when bottom sheet layout has been attached to window
     */
    override fun onAttachedToWindow() {
        super.onAttachedToWindow()
//        addOnLayoutChangeListener(layoutChangedListener)
    }

    /**
     * Called when bottom sheet layout has been detached to window
     */
    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
//        removeOnLayoutChangeListener(layoutChangedListener)
    }

    /**
     * Chane line view style based on collapsed or not
     * @param collapsed Boolean true if collapsed style should be used, false otherwise
     */
    fun toggleLineViewStyle(collapsed: Boolean) {
        StyleCompat.style(lineView).apply(if (collapsed) collapsedLineStyleRes else lineStyleRes)
    }

    /**
     * Change collapsed style of line view
     * @param collapsedLineStyle Int new style
     */
    @Attr(R2.styleable.BandyerSDKDesign_Widget_BottomSheetLayout_bandyer_collapsedLineStyle)
    fun setCollapsedStyle(@AnyRes collapsedLineStyle: Int = 0) {
        this.collapsedLineStyleRes = collapsedLineStyle
    }

    /**
     * Change style of line view in normal state
     * @param lineStyle Int new style
     */
    @Attr(R2.styleable.BandyerSDKDesign_Widget_BottomSheetLayout_bandyer_lineStyle)
    fun setLineStyle(@AnyRes lineStyle: Int = 0) {
        this.lineStyleRes = lineStyle
    }
}


