/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.bottom_sheet

import android.view.View
import androidx.appcompat.app.AppCompatActivity
import com.bandyer.sdk_design.bottom_sheet.items.ActionItem
import com.bandyer.sdk_design.bottom_sheet.items.ActionItemViewHolder
import com.bandyer.sdk_design.bottom_sheet.view.BottomSheetLayoutType
import com.mikepenz.fastadapter.IAdapter
import com.mikepenz.fastadapter.listeners.OnClickListener
import com.mikepenz.fastadapter.select.SelectExtension

/**
 * This class represents a bottom sheet displaying selectable items.
 * @author kristiyan
 */
@Suppress("UNCHECKED_CAST")
open class BandyerSelectableBottomSheet<T : ActionItem>(context: AppCompatActivity,
                                                        selection: Int = -1,
                                                        views: List<T>,
                                                        spanSize: Int,
                                                        peekHeight: Int?,
                                                        bottomSheetLayoutType: BottomSheetLayoutType,
                                                        bottomSheetStyle: Int)
    : BandyerActionBottomSheet<T>(context, views, spanSize, peekHeight, bottomSheetLayoutType, bottomSheetStyle) {

    private var currentItemSelected: ActionItemViewHolder? = null

    init {
        fastAdapter.withSelectable(true)
        fastAdapter.withAllowDeselection(false)
        fastAdapter.withMultiSelect(false)
        fastAdapter.withSelectWithItemUpdate(false)

        fastAdapter.withOnClickListener(object: OnClickListener<ActionItemViewHolder> {
            override fun onClick(v: View?, adapter: IAdapter<ActionItemViewHolder>, item: ActionItemViewHolder, position: Int): Boolean {
                if (!item.isSelected) return false
                currentItemSelected = item
                notifyItemSelected(item, position)
                return true
            }
        })

        if (selection != -1) {
            fastAdapter.getExtension<SelectExtension<ActionItemViewHolder>>(SelectExtension::class.java)?.select(selection)
            currentItemSelected = fastAdapter.getAdapterItem(selection)
        }
    }

    private fun notifyItemSelected(item: ActionItemViewHolder, position: Int) {
        val listener = onActionBottomSheetListener as? OnActionBottomSheetListener<ActionItem, BandyerBottomSheet>
        listener?.onActionClicked(this, item.item, position)
    }

    /**
     * Select the item provided
     * @param actionItem ActionItem to select
     */
    fun selectItem(actionItem: ActionItem?) {
        if (actionItem == null)
            return

        currentItemSelected = ActionItemViewHolder(actionItem)

        fastAdapter.getExtension<SelectExtension<ActionItemViewHolder>>(SelectExtension::class.java)?.deselect()

        val position = fastAdapter.adapterItems.indexOfFirst { it.item == actionItem }
        fastAdapter.getExtension<SelectExtension<ActionItemViewHolder>>(SelectExtension::class.java)?.select(position)
    }
}