/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.bottom_sheet

import android.app.Dialog
import android.content.DialogInterface
import android.os.Build
import android.os.Bundle
import android.os.Handler
import android.os.Looper
import android.view.View
import android.widget.FrameLayout
import androidx.coordinatorlayout.widget.CoordinatorLayout
import androidx.fragment.app.DialogFragment
import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.bottom_sheet.behaviours.BandyerBottomSheetBehaviour
import com.bandyer.sdk_design.bottom_sheet.behaviours.BandyerBottomSheetDialogBehaviour
import com.google.android.material.bottomsheet.BottomSheetBehavior
import com.google.android.material.bottomsheet.BottomSheetDialog
import com.google.android.material.bottomsheet.BottomSheetDialogFragment


/**
 * @suppress
 */
abstract class BandyerBottomSheetDialog : BottomSheetDialogFragment() {

    open val style: Int = R.style.BandyerSDK_DialogStyle

    abstract fun onExpanded()

    abstract fun onCollapsed()

    abstract fun onDialogWillShow()

    abstract fun onSlide(offset: Float)

    abstract fun onStateChanged(@BandyerBottomSheetBehaviour.State newState: Int)

    protected var behavior: BandyerBottomSheetDialogBehaviour<View>? = null

    protected var isUserAction = false

    private var mDismissListener: DialogInterface.OnDismissListener? = null

    private val bottomSheetBehavior = object : BottomSheetBehavior.BottomSheetCallback() {
        override fun onSlide(p0: View, slideOffset: Float) {
            val offset = if (slideOffset < 0) 0f else slideOffset
            onSlide(offset)
        }

        override fun onStateChanged(p0: View, newState: Int) {
            onStateChanged(newState)
            when (newState) {
                BottomSheetBehavior.STATE_EXPANDED -> {
                    isUserAction = false
                    behavior!!.disableDragging = false
                    onExpanded()
                }
                BottomSheetBehavior.STATE_COLLAPSED -> {
                    isUserAction = false
                    behavior!!.disableDragging = false
                    onCollapsed()
                }
                else -> {
                }
            }
        }
    }

    fun setDismissListener(listener: DialogInterface.OnDismissListener) {
        mDismissListener = listener
    }


    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setStyle(DialogFragment.STYLE_NORMAL, style)
    }

    override fun onStart() {
        super.onStart()
        dialog ?: return
        val window = dialog!!.window ?: return
        val decorView = window.decorView
        // fullscreen it
        window.findViewById<View>(R.id.container).fitsSystemWindows = false

        // dark navigation bar icons
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.O) return
        decorView.systemUiVisibility = decorView.systemUiVisibility or
                View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR or
                View.SYSTEM_UI_FLAG_LIGHT_NAVIGATION_BAR
    }

    @Suppress("UNCHECKED_CAST")
    override fun onCreateDialog(savedInstanceState: Bundle?): Dialog {
        return super.onCreateDialog(savedInstanceState).apply {
            setOnShowListener { dialog ->
                setupBottomSheetBehaviour()
                onDialogWillShow()
                isCancelable = false
            }
        }
    }

    private fun setupBottomSheetBehaviour() {
        val bottomSheet = (dialog as BottomSheetDialog).findViewById(R.id.design_bottom_sheet) as FrameLayout?
        val params = bottomSheet!!.layoutParams as CoordinatorLayout.LayoutParams
        params.behavior = BandyerBottomSheetDialogBehaviour<View>(bottomSheet.context, null)
        bottomSheet.layoutParams = params
        behavior = BottomSheetBehavior.from(bottomSheet) as BandyerBottomSheetDialogBehaviour<View>
        behavior!!.state = BottomSheetBehavior.STATE_EXPANDED
        behavior!!.peekHeight = 0
        behavior!!.setBottomSheetCallback(bottomSheetBehavior)
    }

    /**
     * @suppress
     * @see [here](https://medium.com/square-corner-blog/a-small-leak-will-sink-a-great-ship-efbae00f9a0f)
     */
    private fun flushStackLocalLeaks(looper: Looper) {
        val handler = Handler(looper)
        handler.post {
            Looper.myQueue().addIdleHandler {
                handler.sendMessageDelayed(handler.obtainMessage(), 1000)
                true
            }
        }
    }

    override fun dismiss() {
        clean()
        if (isVisible && !isStateSaved) super.dismiss()
    }

    override fun onDismiss(dialog: DialogInterface) {
        super.onDismiss(dialog)
        clean()
    }

    private fun clean() {
        Looper.myLooper()?.let {
            flushStackLocalLeaks(it)
        }
        mDismissListener?.onDismiss(dialog)
        behavior?.setBottomSheetCallback(null)
        behavior = null
        mDismissListener = null
    }

    override fun setCancelable(cancelable: Boolean) {
        dialog ?: return
        val touchOutsideView: View = dialog!!.window!!.decorView.findViewById(R.id.touch_outside)
        behavior?.isHideable = cancelable
        if (cancelable) {
            touchOutsideView.setOnClickListener {
                if (dialog!!.isShowing) dialog!!.cancel()
            }
        } else touchOutsideView.setOnClickListener(null)
    }

    protected fun collapse() {
        isUserAction = true
        behavior?.let {
            it.disableDragging = false
            it.state = BottomSheetBehavior.STATE_COLLAPSED
        }
    }

    protected fun expand() {
        isUserAction = true
        behavior?.let {
            it.disableDragging = false
            it.state = BottomSheetBehavior.STATE_EXPANDED
        }
    }

}