/*
 * Decompiled with CFR 0.152.
 */
package com.azure.resourcemanager.test.policy;

import com.azure.core.http.HttpHeader;
import com.azure.core.http.HttpHeaders;
import com.azure.core.http.HttpPipelineCallContext;
import com.azure.core.http.HttpPipelineNextPolicy;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.HttpResponse;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.util.CoreUtils;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

public final class HttpDebugLoggingPolicy
implements HttpPipelinePolicy {
    private static final ObjectMapper PRETTY_PRINTER = new ObjectMapper().enable(SerializationFeature.INDENT_OUTPUT);
    private static final String REDACTED_PLACEHOLDER = "REDACTED";
    private static final Set<String> DISALLOWED_HEADER_NAMES = new HashSet<String>();
    private static final boolean PRETTY_PRINT_BODY = true;

    public HttpDebugLoggingPolicy() {
        DISALLOWED_HEADER_NAMES.add("authorization");
    }

    public Mono<HttpResponse> process(HttpPipelineCallContext context, HttpPipelineNextPolicy next) {
        Logger logger = LoggerFactory.getLogger((String)context.getData("caller-method").orElse(""));
        long startNs = System.nanoTime();
        return this.logRequest(logger, context.getHttpRequest(), context.getData("requestRetryCount")).then(next.process()).flatMap(response -> this.logResponse(logger, (HttpResponse)response, startNs)).doOnError(throwable -> logger.warn("<-- HTTP FAILED: ", throwable));
    }

    private Mono<Void> logRequest(Logger logger, HttpRequest request, Optional<Object> optionalRetryCount) {
        long contentLength;
        if (!logger.isInfoEnabled()) {
            return Mono.empty();
        }
        StringBuilder requestLogMessage = new StringBuilder();
        requestLogMessage.append("--> ").append(request.getHttpMethod()).append(" ").append(request.getUrl()).append(System.lineSeparator());
        optionalRetryCount.ifPresent(o -> requestLogMessage.append("Try count: ").append(o).append(System.lineSeparator()));
        this.addHeadersToLogMessage(logger, request.getHeaders(), requestLogMessage);
        if (request.getBody() == null) {
            requestLogMessage.append("(empty body)").append(System.lineSeparator()).append("--> END ").append(request.getHttpMethod()).append(System.lineSeparator());
            return this.logAndReturn(logger, requestLogMessage, null);
        }
        String contentType = request.getHeaders().getValue("Content-Type");
        if (this.shouldBodyBeLogged(contentType, contentLength = this.getContentLength(logger, request.getHeaders()))) {
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream((int)contentLength);
            WritableByteChannel bodyContentChannel = Channels.newChannel(outputStream);
            request.setBody(request.getBody().flatMap(byteBuffer -> HttpDebugLoggingPolicy.writeBufferToBodyStream(bodyContentChannel, byteBuffer)).doFinally(ignored -> {
                requestLogMessage.append(contentLength).append("-byte body:").append(System.lineSeparator()).append(this.prettyPrintIfNeeded(logger, contentType, HttpDebugLoggingPolicy.convertStreamToString(outputStream, logger))).append(System.lineSeparator()).append("--> END ").append(request.getHttpMethod()).append(System.lineSeparator());
                logger.info(requestLogMessage.toString());
            }));
            return Mono.empty();
        }
        requestLogMessage.append(contentLength).append("-byte body: (content not logged)").append(System.lineSeparator()).append("--> END ").append(request.getHttpMethod()).append(System.lineSeparator());
        return this.logAndReturn(logger, requestLogMessage, null);
    }

    private Mono<HttpResponse> logResponse(Logger logger, HttpResponse response, long startNs) {
        if (!logger.isInfoEnabled()) {
            return Mono.just((Object)response);
        }
        long tookMs = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNs);
        String contentLengthString = response.getHeaderValue("Content-Length");
        String bodySize = CoreUtils.isNullOrEmpty((CharSequence)contentLengthString) ? "unknown-length body" : contentLengthString + "-byte body";
        StringBuilder responseLogMessage = new StringBuilder();
        responseLogMessage.append("<-- ").append(response.getStatusCode()).append(" ").append(response.getRequest().getUrl()).append(" (").append(tookMs).append(" ms, ").append(bodySize).append(")").append(System.lineSeparator());
        this.addHeadersToLogMessage(logger, response.getHeaders(), responseLogMessage);
        String contentTypeHeader = response.getHeaderValue("Content-Type");
        long contentLength = this.getContentLength(logger, response.getHeaders());
        if (this.shouldBodyBeLogged(contentTypeHeader, contentLength)) {
            HttpResponse bufferedResponse = response.buffer();
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream((int)contentLength);
            WritableByteChannel bodyContentChannel = Channels.newChannel(outputStream);
            return bufferedResponse.getBody().flatMap(byteBuffer -> HttpDebugLoggingPolicy.writeBufferToBodyStream(bodyContentChannel, byteBuffer)).doFinally(ignored -> {
                responseLogMessage.append("Response body:").append(System.lineSeparator()).append(this.prettyPrintIfNeeded(logger, contentTypeHeader, HttpDebugLoggingPolicy.convertStreamToString(outputStream, logger))).append(System.lineSeparator()).append("<-- END HTTP");
                logger.info(responseLogMessage.toString());
            }).then(Mono.just((Object)bufferedResponse));
        }
        responseLogMessage.append("(body content not logged)").append(System.lineSeparator()).append("<-- END HTTP");
        return this.logAndReturn(logger, responseLogMessage, response);
    }

    private <T> Mono<T> logAndReturn(Logger logger, StringBuilder logMessageBuilder, T data) {
        logger.info(logMessageBuilder.toString());
        return Mono.justOrEmpty(data);
    }

    private void addHeadersToLogMessage(Logger logger, HttpHeaders headers, StringBuilder sb) {
        for (HttpHeader header : headers) {
            String headerName = header.getName();
            sb.append(headerName).append(":");
            if (!DISALLOWED_HEADER_NAMES.contains(headerName.toLowerCase(Locale.ROOT))) {
                sb.append(header.getValue());
            } else {
                sb.append(REDACTED_PLACEHOLDER);
            }
            sb.append(System.lineSeparator());
        }
    }

    private String prettyPrintIfNeeded(Logger logger, String contentType, String body) {
        String result = body;
        if (contentType != null && (contentType.startsWith("application/json") || contentType.startsWith("text/json"))) {
            try {
                JsonNode deserialized = PRETTY_PRINTER.readTree(body);
                result = PRETTY_PRINTER.writeValueAsString((Object)deserialized);
            }
            catch (Exception e) {
                logger.warn("Failed to pretty print JSON: {}", (Object)e.getMessage());
            }
        }
        return result;
    }

    private long getContentLength(Logger logger, HttpHeaders headers) {
        long contentLength = 0L;
        String contentLengthString = headers.getValue("Content-Length");
        if (CoreUtils.isNullOrEmpty((CharSequence)contentLengthString)) {
            return contentLength;
        }
        try {
            contentLength = Long.parseLong(contentLengthString);
        }
        catch (NullPointerException | NumberFormatException e) {
            logger.warn("Could not parse the HTTP header content-length: '{}'.", (Object)headers.getValue("content-length"), (Object)e);
        }
        return contentLength;
    }

    private boolean shouldBodyBeLogged(String contentTypeHeader, long contentLength) {
        return !"application/octet-stream".equalsIgnoreCase(contentTypeHeader) && contentLength != 0L;
    }

    private static String convertStreamToString(ByteArrayOutputStream stream, Logger logger) {
        try {
            return stream.toString(StandardCharsets.UTF_8.name());
        }
        catch (UnsupportedEncodingException ex) {
            logger.error(ex.toString());
            throw new RuntimeException(ex);
        }
    }

    private static Mono<ByteBuffer> writeBufferToBodyStream(WritableByteChannel channel, ByteBuffer byteBuffer) {
        try {
            channel.write(byteBuffer.duplicate());
            return Mono.just((Object)byteBuffer);
        }
        catch (IOException ex) {
            return Mono.error((Throwable)ex);
        }
    }
}

