/*
 * Decompiled with CFR 0.152.
 */
package com.azure.communication.common;

import com.azure.communication.common.CommunicationTokenRefreshOptions;
import com.azure.communication.common.EntraCommunicationTokenCredentialOptions;
import com.azure.communication.common.EntraTokenCredential;
import com.azure.communication.common.implementation.TokenParser;
import com.azure.core.credential.AccessToken;
import com.azure.core.util.FluxUtil;
import com.azure.core.util.logging.ClientLogger;
import java.io.IOException;
import java.time.OffsetDateTime;
import java.util.Date;
import java.util.Objects;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import reactor.core.publisher.Mono;

public final class CommunicationTokenCredential
implements AutoCloseable {
    private static final int DEFAULT_EXPIRING_OFFSET_MINUTES = 10;
    private static final int DEFAULT_REFRESH_AFTER_TTL_DIVIDER = 2;
    private final ClientLogger logger = new ClientLogger(CommunicationTokenCredential.class);
    private AccessToken accessToken;
    private final TokenParser tokenParser = new TokenParser();
    private Supplier<Mono<String>> refresher;
    private FetchingTask fetchingTask;
    private boolean isClosed = false;
    private boolean isEntra = false;

    public CommunicationTokenCredential(String token) {
        Objects.requireNonNull(token, "'token' cannot be null.");
        this.setToken(token);
    }

    public CommunicationTokenCredential(CommunicationTokenRefreshOptions tokenRefreshOptions) {
        Supplier<String> tokenRefresher = tokenRefreshOptions.getTokenRefresherSync();
        this.refresher = tokenRefresher != null ? () -> Mono.fromSupplier((Supplier)tokenRefresher) : tokenRefreshOptions.getTokenRefresher();
        Objects.requireNonNull(this.refresher, "'tokenRefresher' cannot be null.");
        if (tokenRefreshOptions.getInitialToken() != null) {
            this.setToken(tokenRefreshOptions.getInitialToken());
        }
        if (tokenRefreshOptions.isRefreshProactively()) {
            this.scheduleRefresher();
        }
    }

    public CommunicationTokenCredential(EntraCommunicationTokenCredentialOptions entraTokenOptions) {
        Objects.requireNonNull(entraTokenOptions, "'entraTokenOptions' cannot be null.");
        EntraTokenCredential entraTokenCredential = new EntraTokenCredential(entraTokenOptions);
        this.refresher = entraTokenCredential::exchangeEntraToken;
        this.isEntra = true;
    }

    private void scheduleRefresher() {
        OffsetDateTime nextFetchTime;
        if (this.isTokenExpired(this.accessToken)) {
            nextFetchTime = OffsetDateTime.now();
        } else {
            OffsetDateTime now = OffsetDateTime.now();
            long tokenTtlMs = this.accessToken.getExpiresAt().toInstant().toEpochMilli() - now.toInstant().toEpochMilli();
            long nextFetchTimeMs = this.isTokenExpiringSoon() ? tokenTtlMs / 2L : tokenTtlMs - TimeUnit.MILLISECONDS.convert(10L, TimeUnit.MINUTES);
            nextFetchTime = now.plusNanos(TimeUnit.NANOSECONDS.convert(nextFetchTimeMs, TimeUnit.MILLISECONDS));
        }
        this.fetchingTask = new FetchingTask(this, nextFetchTime);
    }

    private boolean isTokenExpired(AccessToken accessToken) {
        return this.isEntra || accessToken == null || accessToken.isExpired();
    }

    private boolean isTokenExpiringSoon() {
        return this.accessToken == null || OffsetDateTime.now().isAfter(this.accessToken.getExpiresAt().minusMinutes(10L));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Mono<AccessToken> getToken() {
        if (this.isClosed) {
            return FluxUtil.monoError((ClientLogger)this.logger, (RuntimeException)new RuntimeException("getToken called on closed CommunicationTokenCredential object"));
        }
        if (this.isTokenExpired(this.accessToken) && this.refresher != null) {
            CommunicationTokenCredential communicationTokenCredential = this;
            synchronized (communicationTokenCredential) {
                if (this.isTokenExpired(this.accessToken) && this.refresher != null) {
                    return this.fetchFreshToken().flatMap(token -> {
                        this.accessToken = this.tokenParser.parseJWTToken((String)token);
                        if (this.isTokenExpired(this.accessToken)) {
                            return FluxUtil.monoError((ClientLogger)this.logger, (RuntimeException)new IllegalArgumentException("The token returned from the tokenRefresher is expired."));
                        }
                        return Mono.just((Object)this.accessToken);
                    });
                }
            }
        }
        return Mono.just((Object)this.accessToken);
    }

    @Override
    public void close() throws IOException {
        this.isClosed = true;
        if (this.fetchingTask != null) {
            this.fetchingTask.stopTimer();
            this.fetchingTask = null;
        }
        this.refresher = null;
        this.isEntra = false;
    }

    boolean hasProactiveFetcher() {
        return this.fetchingTask != null;
    }

    private void setToken(String freshToken) {
        this.accessToken = this.tokenParser.parseJWTToken(freshToken);
        if (this.hasProactiveFetcher()) {
            this.scheduleRefresher();
        }
    }

    private Mono<String> fetchFreshToken() {
        Mono<String> tokenAsync = this.refresher.get();
        if (tokenAsync == null) {
            return FluxUtil.monoError((ClientLogger)this.logger, (RuntimeException)new RuntimeException("get() function of the token refresher should not return null."));
        }
        return tokenAsync;
    }

    private static class FetchingTask {
        private final CommunicationTokenCredential host;
        private Timer expiringTimer;
        private OffsetDateTime nextFetchTime;

        FetchingTask(CommunicationTokenCredential tokenHost, OffsetDateTime nextFetchAt) {
            this.host = tokenHost;
            this.nextFetchTime = nextFetchAt;
            this.stopTimer();
            this.startTimer();
        }

        private synchronized void startTimer() {
            this.expiringTimer = new Timer();
            Date expiring = Date.from(this.nextFetchTime.toInstant());
            this.expiringTimer.schedule((TimerTask)new TokenExpiringTask(this), expiring);
        }

        private synchronized void stopTimer() {
            if (this.expiringTimer == null) {
                return;
            }
            this.expiringTimer.cancel();
            this.expiringTimer.purge();
            this.expiringTimer = null;
        }

        private Mono<String> fetchFreshToken() {
            return this.host.fetchFreshToken();
        }

        private void setToken(String freshTokenString) {
            this.host.setToken(freshTokenString);
        }

        private boolean isTokenExpired(String freshTokenString) {
            return this.host.tokenParser.parseJWTToken(freshTokenString).isExpired();
        }

        private static class TokenExpiringTask
        extends TimerTask {
            private final ClientLogger logger = new ClientLogger(TokenExpiringTask.class);
            private final FetchingTask tokenCache;

            TokenExpiringTask(FetchingTask host) {
                this.tokenCache = host;
            }

            @Override
            public void run() {
                try {
                    Mono tokenAsync = this.tokenCache.fetchFreshToken();
                    tokenAsync.subscribe(token -> {
                        if (!this.tokenCache.isTokenExpired(token)) {
                            this.tokenCache.setToken(token);
                        } else {
                            this.logger.logExceptionAsError((RuntimeException)new IllegalArgumentException("The token returned from the tokenRefresher is expired."));
                        }
                    });
                }
                catch (Exception exception) {
                    this.logger.logExceptionAsError(new RuntimeException(exception));
                }
            }
        }
    }
}

