/*
 * Decompiled with CFR 0.152.
 */
package com.atomikos.recovery.xa;

import com.atomikos.datasource.xa.RecoveryScan;
import com.atomikos.datasource.xa.XAExceptionHelper;
import com.atomikos.datasource.xa.XID;
import com.atomikos.icatch.config.Configuration;
import com.atomikos.icatch.event.Event;
import com.atomikos.icatch.event.transaction.ParticipantHeuristicEvent;
import com.atomikos.logging.Logger;
import com.atomikos.logging.LoggerFactory;
import com.atomikos.publish.EventPublisher;
import com.atomikos.recovery.LogException;
import com.atomikos.recovery.LogReadException;
import com.atomikos.recovery.PendingTransactionRecord;
import com.atomikos.recovery.TxState;
import com.atomikos.recovery.xa.InMemoryPreviousXidRepository;
import com.atomikos.recovery.xa.PreviousXidRepository;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.transaction.xa.XAException;
import javax.transaction.xa.XAResource;

public class XARecoveryManager {
    private static final Logger LOGGER = LoggerFactory.createLogger(XARecoveryManager.class);
    private static XARecoveryManager instance;
    private RecoveryScan.XidSelector xidSelector;
    private String tmUniqueName;
    private long maxTimeout;
    private Map<String, PreviousXidRepository> previousXidRepositoryMap = new HashMap<String, PreviousXidRepository>();

    public XARecoveryManager(final String tmUniqueName) {
        this.tmUniqueName = tmUniqueName;
        this.xidSelector = new RecoveryScan.XidSelector(){

            @Override
            public boolean selects(XID xid) {
                boolean ret = false;
                String branch = xid.getBranchQualifierAsString();
                if (branch.startsWith(tmUniqueName)) {
                    ret = true;
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.logDebug(this + ": recovering XID: " + xid);
                    }
                } else if (LOGGER.isDebugEnabled()) {
                    LOGGER.logDebug(this + ": XID " + xid + " with branch " + branch + " is not under my responsibility");
                }
                return ret;
            }

            public String toString() {
                return XARecoveryManager.this.toString();
            }
        };
        this.maxTimeout = Configuration.getConfigProperties().getMaxTimeout();
    }

    public static XARecoveryManager getInstance() {
        return instance;
    }

    public static void installXARecoveryManager(String tmUniqueName) {
        instance = tmUniqueName == null ? null : new XARecoveryManager(tmUniqueName);
    }

    public boolean recover(XAResource xaResource, long startOfRecoveryScan, Collection<PendingTransactionRecord> expiredCommittingCoordinators, Collection<PendingTransactionRecord> indoubtForeignCoordinatorsToKeep, String uniqueResourceName) throws XAException, LogReadException {
        List<XID> xidsToRecover = this.retrievePreparedXidsFromXaResource(xaResource);
        PreviousXidRepository previousXidRepository = this.getPreviousXidRepository(uniqueResourceName);
        boolean success = this.recoverXids(xidsToRecover, previousXidRepository, expiredCommittingCoordinators, indoubtForeignCoordinatorsToKeep, xaResource, startOfRecoveryScan);
        previousXidRepository.forgetXidsExpiredAt(startOfRecoveryScan);
        return success;
    }

    private PreviousXidRepository getPreviousXidRepository(String uniqueResourceName) {
        PreviousXidRepository ret = this.previousXidRepositoryMap.get(uniqueResourceName);
        if (ret == null) {
            ret = new InMemoryPreviousXidRepository();
            this.previousXidRepositoryMap.put(uniqueResourceName, ret);
        }
        return ret;
    }

    private boolean recoverXids(List<XID> xidsToRecover, PreviousXidRepository previousXidRepository, Collection<PendingTransactionRecord> expiredCommittingCoordinators, Collection<PendingTransactionRecord> indoubtForeignCoordinatorsToKeep, XAResource xaResource, long startOfRecoveryScan) {
        boolean allExpiredCommitsDone = true;
        long xidDetectionTime = System.currentTimeMillis();
        List<XID> expiredPreviousXids = previousXidRepository.findXidsExpiredAt(startOfRecoveryScan);
        Collection expiredCommittingCoordinatorIds = PendingTransactionRecord.extractCoordinatorIds(expiredCommittingCoordinators, (TxState[])new TxState[]{TxState.COMMITTING, TxState.IN_DOUBT});
        Collection foreignIndoubtCoordinatorIds = PendingTransactionRecord.extractCoordinatorIds(indoubtForeignCoordinatorsToKeep, (TxState[])new TxState[]{TxState.IN_DOUBT});
        for (XID xid : xidsToRecover) {
            String coordinatorId = xid.getGlobalTransactionIdAsString();
            if (expiredCommittingCoordinatorIds.contains(coordinatorId)) {
                allExpiredCommitsDone = allExpiredCommitsDone && this.replayCommit(xid, xaResource);
                continue;
            }
            if (expiredPreviousXids.contains(xid)) {
                if (!foreignIndoubtCoordinatorIds.contains(coordinatorId) && this.presumedAbort(xid, xaResource)) continue;
                previousXidRepository.remember(xid, startOfRecoveryScan + 1L);
                continue;
            }
            previousXidRepository.remember(xid, xidDetectionTime + this.maxTimeout);
        }
        return allExpiredCommitsDone;
    }

    private boolean replayCommit(XID xid, XAResource xaResource) {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.logDebug(this + ": replaying commit of xid: " + xid);
        }
        boolean forgetInLog = false;
        try {
            xaResource.commit(xid, false);
            forgetInLog = true;
        }
        catch (XAException e) {
            if (this.alreadyHeuristicallyTerminatedByResource(e)) {
                forgetInLog = this.handleHeuristicTerminationByResource(xid, xaResource, e, true);
            }
            if (this.xidTerminatedInResourceByConcurrentCommit(e)) {
                forgetInLog = true;
            }
            LOGGER.logWarning(XAExceptionHelper.formatLogMessage("Transient error while replaying commit", e, "will retry later"));
        }
        return forgetInLog;
    }

    private boolean alreadyHeuristicallyTerminatedByResource(XAException e) {
        boolean ret = false;
        switch (e.errorCode) {
            case 5: 
            case 6: 
            case 7: 
            case 8: {
                ret = true;
            }
        }
        return ret;
    }

    private boolean xidTerminatedInResourceByConcurrentCommit(XAException e) {
        return this.xidNoLongerKnownByResource(e);
    }

    private boolean xidTerminatedInResourceByConcurrentRollback(XAException e) {
        return this.xidNoLongerKnownByResource(e);
    }

    private boolean xidNoLongerKnownByResource(XAException e) {
        boolean ret = false;
        switch (e.errorCode) {
            case -5: 
            case -4: {
                ret = true;
            }
        }
        return ret;
    }

    private boolean handleHeuristicTerminationByResource(XID xid, XAResource xaResource, XAException e, boolean commitDesired) {
        boolean forgetInLog = true;
        try {
            this.notifyLogOfHeuristic(xid, e, commitDesired);
            if (e.errorCode != 8) {
                this.forgetXidInXaResource(xid, xaResource);
            } else {
                forgetInLog = false;
            }
        }
        catch (LogException transientLogWriteException) {
            LOGGER.logWarning("Failed to log heuristic termination of Xid: " + xid + " - ignoring to retry later", (Throwable)transientLogWriteException);
        }
        return forgetInLog;
    }

    private void forgetXidInXaResource(XID xid, XAResource xaResource) {
        try {
            xaResource.forget(xid);
        }
        catch (XAException e) {
            LOGGER.logWarning(XAExceptionHelper.formatLogMessage("Unexpected error during forget", e, "ignoring"));
        }
    }

    private void notifyLogOfHeuristic(XID xid, XAException e, boolean commitDesired) throws LogException {
        switch (e.errorCode) {
            case 8: {
                this.fireTransactionHeuristicEvent(xid, TxState.HEUR_HAZARD);
                break;
            }
            case 7: {
                if (commitDesired) break;
                this.fireTransactionHeuristicEvent(xid, TxState.HEUR_COMMITTED);
                break;
            }
            case 5: {
                this.fireTransactionHeuristicEvent(xid, TxState.HEUR_MIXED);
                break;
            }
            case 6: {
                if (!commitDesired) break;
                this.fireTransactionHeuristicEvent(xid, TxState.HEUR_ABORTED);
                break;
            }
        }
    }

    private void fireTransactionHeuristicEvent(XID xid, TxState state) {
        ParticipantHeuristicEvent event = new ParticipantHeuristicEvent(xid.getGlobalTransactionIdAsString(), xid.toString(), state);
        EventPublisher.INSTANCE.publish((Event)event);
    }

    private boolean presumedAbort(XID xid, XAResource xaResource) {
        boolean ret = false;
        if (LOGGER.isDebugEnabled()) {
            LOGGER.logDebug(this + ": presumed abort of xid: " + xid);
        }
        try {
            xaResource.rollback(xid);
            ret = true;
        }
        catch (XAException e) {
            if (this.alreadyHeuristicallyTerminatedByResource(e)) {
                ret = this.handleHeuristicTerminationByResource(xid, xaResource, e, false);
            }
            if (this.xidTerminatedInResourceByConcurrentRollback(e)) {
                ret = true;
            }
            LOGGER.logWarning(XAExceptionHelper.formatLogMessage("Unexpected exception during recovery", e, "ignoring to retry later"));
        }
        return ret;
    }

    private List<XID> retrievePreparedXidsFromXaResource(XAResource xaResource) throws XAException {
        ArrayList<XID> ret = new ArrayList();
        try {
            ret = RecoveryScan.recoverXids(xaResource, this.xidSelector);
        }
        catch (XAException e) {
            LOGGER.logWarning(XAExceptionHelper.formatLogMessage("Error while retrieving xids from resource", e, "will retry later"));
            throw e;
        }
        return ret;
    }

    public String toString() {
        return "XARecoveryManager " + this.tmUniqueName;
    }

    public boolean hasPendingXids(String uniqueResourceName) {
        PreviousXidRepository previousXidRepository = this.getPreviousXidRepository(uniqueResourceName);
        return !previousXidRepository.isEmpty();
    }
}

