package com.atlassian.velocity.htmlsafe.util;

/**
 * Utility to perform checks on parameters.
 */
public final class Check {
    /**
     * Check that {@code reference} is not {@code null}. If it is, throw a
     * {@code NullPointerException}.
     *
     * @param reference reference to check is {@code null} or not
     * @param <T>       reference to check is {@code null} or not
     * @return {@code reference} so it may be used
     * @throws NullPointerException if {@code reference} is {@code null}
     */
    public static <T> T notNull(T reference) {
        if (reference == null) {
            throw new NullPointerException();
        }
        return reference;
    }

    /**
     * Check that {@code reference} is not {@code null}. If it is, throw a
     * {@code NullPointerException}.
     *
     * @param reference    reference to check is {@code null} or not
     * @param <T>          reference to check is {@code null} or not
     * @param errorMessage message passed to the {@code NullPointerException} constructor
     *                     to give more context when debugging
     * @return {@code reference} so it may be used
     * @throws NullPointerException if {@code reference} is {@code null}
     */
    public static <T> T notNull(T reference, Object errorMessage) {
        if (reference == null) {
            throw new NullPointerException(String.valueOf(errorMessage));
        }
        return reference;
    }

    /**
     * Ensures the truth of an expression involving one or more parameters to
     * the calling method.
     *
     * @param expression a boolean expression
     * @throws IllegalArgumentException if {@code expression} is false
     */
    public static void argument(boolean expression) {
        if (!expression) {
            throw new IllegalArgumentException();
        }
    }

    /**
     * Ensures the truth of an expression involving one or more parameters to
     * the calling method.
     *
     * @param expression   a boolean expression
     * @param errorMessage the exception message to use if the check fails; will be
     *                     converted to a string using {@link String#valueOf(Object)}
     * @throws IllegalArgumentException if {@code expression} is false
     */
    public static void argument(boolean expression, Object errorMessage) {
        if (!expression) {
            throw new IllegalArgumentException(String.valueOf(errorMessage));
        }
    }
}
