package com.atlassian.vcache.internal.memcached;

import com.atlassian.vcache.DirectExternalCache;
import com.atlassian.vcache.ExternalCacheSettings;
import com.atlassian.vcache.JvmCache;
import com.atlassian.vcache.JvmCacheSettings;
import com.atlassian.vcache.Marshaller;
import com.atlassian.vcache.StableReadExternalCache;
import com.atlassian.vcache.TransactionalExternalCache;
import com.atlassian.vcache.internal.BegunTransactionalActivityHandler;
import com.atlassian.vcache.internal.RequestContext;
import com.atlassian.vcache.internal.VCacheCreationHandler;
import com.atlassian.vcache.internal.VCacheSettingsDefaultsProvider;
import com.atlassian.vcache.internal.core.ExternalCacheKeyGenerator;
import com.atlassian.vcache.internal.core.metrics.MetricsCollector;
import com.atlassian.vcache.internal.core.service.AbstractVCacheService;
import com.atlassian.vcache.internal.core.service.GuavaJvmCache;
import com.atlassian.vcache.internal.guava.GuavaDirectExternalCache;
import com.atlassian.vcache.internal.guava.GuavaStableReadExternalCache;
import com.atlassian.vcache.internal.guava.GuavaTransactionalExternalCache;
import com.atlassian.vcache.internal.guava.GuavaUtils;
import net.spy.memcached.MemcachedClientIF;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * Main service that is backed by <tt>Memcached</tt>.
 */
public class MemcachedVCacheService extends AbstractVCacheService {
    private static final Logger log = LoggerFactory.getLogger(MemcachedVCacheService.class);

    private final Supplier<MemcachedClientIF> clientSupplier;
    private final Function<String, Boolean> dontExternaliseCache;

    /**
     * Creates an instance, with the default {@link com.atlassian.vcache.internal.core.ExternalCacheKeyGenerator}.
     *
     * @param productIdentifier the unique product identifier, such as <tt>jira</tt> or <tt>confluence</tt>
     * @param clientSupplier    the supplier for the Memcached client
     * @param contextSupplier   the supplier for the request context.
     * @param defaultsProvider  the cache defaults provider
     * @param creationHandler   the handler called when caches are being created.
     * @param metricsCollector  the collector for gathering metrics.
     * @deprecated since 1.3.0, use {@link #MemcachedVCacheService(MemcachedVCacheServiceSettings)}
     */
    @Deprecated
    public MemcachedVCacheService(String productIdentifier,
                                  Supplier<MemcachedClientIF> clientSupplier,
                                  Supplier<RequestContext> contextSupplier,
                                  VCacheSettingsDefaultsProvider defaultsProvider,
                                  VCacheCreationHandler creationHandler,
                                  MetricsCollector metricsCollector,
                                  BegunTransactionalActivityHandler begunTransactionalActivityHandler) {
        this(new MemcachedVCacheServiceSettingsBuilder()
                .productIdentifier(productIdentifier)
                .clientSupplier(clientSupplier)
                .defaultsProvider(defaultsProvider)
                .contextSupplier(contextSupplier)
                .creationHandler(creationHandler)
                .metricsCollector(metricsCollector)
                .begunTransactionalActivityHandler(begunTransactionalActivityHandler)
                .build());
    }

    /**
     * Creates an instance, with the supplied {@link com.atlassian.vcache.internal.core.ExternalCacheKeyGenerator}.
     *
     * @param clientSupplier            the supplier for the Memcached client
     * @param contextSupplier           the supplier for the request context.
     * @param defaultsProvider          the cache defaults provider
     * @param creationHandler           the handler called when caches are being created.
     * @param metricsCollector          the collector for gathering metrics.
     * @param externalCacheKeyGenerator the generator of external cache keys.
     * @deprecated since 1.3.0, use {@link #MemcachedVCacheService(MemcachedVCacheServiceSettings)}
     */
    @Deprecated
    public MemcachedVCacheService(Supplier<MemcachedClientIF> clientSupplier,
                                  Supplier<RequestContext> contextSupplier,
                                  VCacheSettingsDefaultsProvider defaultsProvider,
                                  VCacheCreationHandler creationHandler,
                                  MetricsCollector metricsCollector,
                                  ExternalCacheKeyGenerator externalCacheKeyGenerator,
                                  BegunTransactionalActivityHandler begunTransactionalActivityHandler) {
        this(new MemcachedVCacheServiceSettingsBuilder()
                .clientSupplier(clientSupplier)
                .contextSupplier(contextSupplier)
                .defaultsProvider(defaultsProvider)
                .creationHandler(creationHandler)
                .metricsCollector(metricsCollector)
                .externalCacheKeyGenerator(externalCacheKeyGenerator)
                .begunTransactionalActivityHandler(begunTransactionalActivityHandler)
                .build());
    }

    /**
     * Creates an instance using the supplied settings.
     *
     * @param settings the settings to be used for creating the instance.
     * @since 1.3.0
     */
    public MemcachedVCacheService(MemcachedVCacheServiceSettings settings) {
        super(settings.getContextSupplier(),
                settings.getDefaultsProvider(),
                settings.getCreationHandler(),
                settings.getMetricsCollector(),
                settings.getExternalCacheKeyGenerator(),
                settings.getBegunTransactionalActivityHandler());
        this.clientSupplier = requireNonNull(settings.getClientSupplier());
        this.dontExternaliseCache = requireNonNull(settings.getDontExternaliseCache());
    }

    @Nonnull
    @Override
    protected Logger log() {
        return log;
    }

    @Nonnull
    @Override
    protected <K, V> JvmCache<K, V> createJvmCache(String name, JvmCacheSettings settings) {
        return new GuavaJvmCache<>(name, settings);
    }

    @Nonnull
    @Override
    protected <V> TransactionalExternalCache<V> createTransactionalExternalCache(String name,
                                                                                 ExternalCacheSettings settings,
                                                                                 Marshaller<V> valueMarshaller,
                                                                                 boolean valueSerializable) {
        if (dontExternaliseCache.apply(name)) {
            log.trace("Cache {}: not being externalised", name);
            return new GuavaTransactionalExternalCache<>(
                    name,
                    GuavaUtils.buildDelegate(settings),
                    contextSupplier,
                    externalCacheKeyGenerator,
                    Optional.of(valueMarshaller),
                    transactionControlManager);
        }

        return new MemcachedTransactionalExternalCache<>(
                clientSupplier,
                contextSupplier,
                externalCacheKeyGenerator,
                name,
                valueMarshaller,
                settings,
                transactionControlManager);
    }

    @Nonnull
    @Override
    protected <V> StableReadExternalCache<V> createStableReadExternalCache(String name,
                                                                           ExternalCacheSettings settings,
                                                                           Marshaller<V> valueMarshaller,
                                                                           boolean valueSerializable) {
        if (dontExternaliseCache.apply(name)) {
            log.trace("Cache {}: not being externalised", name);
            return new GuavaStableReadExternalCache<>(
                    name,
                    GuavaUtils.buildDelegate(settings),
                    contextSupplier,
                    externalCacheKeyGenerator,
                    Optional.of(valueMarshaller));

        }

        return new MemcachedStableReadExternalCache<>(
                clientSupplier,
                contextSupplier,
                externalCacheKeyGenerator,
                name,
                valueMarshaller,
                settings);
    }

    @Nonnull
    @Override
    protected <V> DirectExternalCache<V> createDirectExternalCache(String name,
                                                                   ExternalCacheSettings settings,
                                                                   Marshaller<V> valueMarshaller,
                                                                   boolean valueSerializable) {
        if (dontExternaliseCache.apply(name)) {
            log.trace("Cache {}: not being externalised", name);
            return new GuavaDirectExternalCache<>(
                    name,
                    GuavaUtils.buildDelegate(settings),
                    contextSupplier,
                    externalCacheKeyGenerator,
                    Optional.of(valueMarshaller));
        }
        return new MemcachedDirectExternalCache<>(
                clientSupplier,
                contextSupplier,
                externalCacheKeyGenerator,
                name,
                valueMarshaller,
                settings);
    }
}
