package com.atlassian.vcache.internal.memcached;

import com.atlassian.vcache.ExternalCacheException;
import com.atlassian.vcache.ExternalCacheSettings;
import com.atlassian.vcache.Marshaller;
import com.atlassian.vcache.MarshallerException;
import com.atlassian.vcache.PutPolicy;
import com.atlassian.vcache.internal.RequestContext;
import com.atlassian.vcache.internal.core.ExternalCacheKeyGenerator;
import com.atlassian.vcache.internal.core.VCacheCoreUtils;
import com.atlassian.vcache.internal.core.service.AbstractStableReadExternalCache;
import com.atlassian.vcache.internal.core.service.VersionedExternalCacheRequestContext;
import net.spy.memcached.MemcachedClientIF;
import net.spy.memcached.OperationTimeoutException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import static com.atlassian.vcache.internal.core.VCacheCoreUtils.isEmpty;
import static com.atlassian.vcache.internal.core.VCacheCoreUtils.marshall;
import static com.atlassian.vcache.internal.core.VCacheCoreUtils.unmarshall;
import static com.atlassian.vcache.internal.memcached.MemcachedUtils.expiryTime;
import static com.atlassian.vcache.internal.memcached.MemcachedUtils.putOperationForPolicy;
import static java.util.Objects.requireNonNull;

/**
 * Implementation that backs onto Memcached.
 *
 * @param <V> the value type
 * @since 1.0.0
 */
class MemcachedStableReadExternalCache<V>
        extends AbstractStableReadExternalCache<V> {
    private static final Logger log = LoggerFactory.getLogger(MemcachedStableReadExternalCache.class);

    private final Supplier<MemcachedClientIF> clientSupplier;
    private final Supplier<RequestContext> contextSupplier;
    private final ExternalCacheKeyGenerator keyGenerator;
    private final Marshaller<V> valueMarshaller;
    private final int ttlSeconds;

    MemcachedStableReadExternalCache(
            Supplier<MemcachedClientIF> clientSupplier,
            Supplier<RequestContext> contextSupplier,
            ExternalCacheKeyGenerator keyGenerator,
            String name,
            Marshaller<V> valueMarshaller,
            ExternalCacheSettings settings) {
        super(name);
        this.clientSupplier = requireNonNull(clientSupplier);
        this.contextSupplier = requireNonNull(contextSupplier);
        this.keyGenerator = requireNonNull(keyGenerator);
        this.valueMarshaller = requireNonNull(valueMarshaller);
        this.ttlSeconds = VCacheCoreUtils.roundUpToSeconds(settings.getDefaultTtl().get());
    }

    @Nonnull
    @Override
    public CompletionStage<Boolean> put(String internalKey, V value, PutPolicy policy) {
        final VersionedExternalCacheRequestContext<V> cacheContext = ensureCacheContext();
        return perform(
                () -> {
                    final String externalKey = cacheContext.externalEntryKeyFor(internalKey);
                    final byte[] valueBytes = valueMarshaller.marshall(value);

                    final Future<Boolean> putOp =
                            putOperationForPolicy(policy, clientSupplier.get(), externalKey, expiryTime(ttlSeconds), valueBytes);

                    return putOp.get();
                },
                (result) -> {
                    if (result) {
                        cacheContext.recordValue(internalKey, Optional.of(value));
                    } else {
                        cacheContext.forgetValue(internalKey);
                    }
                });
    }

    @Nonnull
    @Override
    public CompletionStage<Void> remove(Iterable<String> internalKeys) {
        // There is no bulk delete in the api, so need to remove each one async
        return perform(() -> {
            if (isEmpty(internalKeys)) {
                return null;
            }

            // Lodge all the requests for delete
            final VersionedExternalCacheRequestContext<V> cacheContext = ensureCacheContext();
            final Map<String, Future<Boolean>> deleteOps =
                    StreamSupport.stream(internalKeys.spliterator(), false)
                            .distinct()
                            .collect(Collectors.toMap(
                                    k -> k,
                                    k -> clientSupplier.get().delete(cacheContext.externalEntryKeyFor(k))
                            ));

            // Need to loop verifying the outcome. If an exception was thrown for an operation,
            // then do not record the removal.
            Exception failureException = null;
            for (Map.Entry<String, Future<Boolean>> delOp : deleteOps.entrySet()) {
                try {
                    delOp.getValue().get();
                    cacheContext.recordValue(delOp.getKey(), Optional.empty());
                } catch (ExecutionException | InterruptedException ex) {
                    log.info("Cache {}: unable to remove key {}", name, delOp.getKey(), ex);
                    failureException = ex;
                }
            }

            // Finally, if there was a failure, then re-throw the last one reported (as good as any!)
            if (failureException != null) {
                if (failureException instanceof ExecutionException) {
                    throw (ExecutionException) failureException;
                }
                throw (InterruptedException) failureException;
            }

            return null;
        });
    }

    @Nonnull
    @Override
    public CompletionStage<Void> removeAll() {
        return perform(() -> {
            final VersionedExternalCacheRequestContext<V> cacheContext = ensureCacheContext();
            cacheContext.updateCacheVersion(
                    MemcachedUtils.incrementCacheVersion(clientSupplier, cacheContext.externalCacheVersionKey()));
            cacheContext.forgetAllValues();
            return null;
        });
    }

    @Nonnull
    @Override
    protected Logger getLogger() {
        return log;
    }

    @Nonnull
    @Override
    protected VersionedExternalCacheRequestContext<V> ensureCacheContext() throws OperationTimeoutException {
        final RequestContext requestContext = contextSupplier.get();

        return requestContext.computeIfAbsent(this, () -> {
            // Need to build a new context, which involves getting the current cache version, or setting it if it does
            // not exist.
            log.trace("Cache {}: Setting up a new context", name);
            final VersionedExternalCacheRequestContext<V> newCacheContext = new VersionedExternalCacheRequestContext<>(
                    keyGenerator, name, requestContext::partitionIdentifier);
            newCacheContext.updateCacheVersion(
                    MemcachedUtils.obtainCacheVersion(clientSupplier, newCacheContext.externalCacheVersionKey()));
            return newCacheContext;
        });
    }

    @Nonnull
    @Override
    protected V handleCreation(String internalKey, Supplier<V> supplier)
            throws MarshallerException, ExecutionException, InterruptedException {
        final VersionedExternalCacheRequestContext<V> cacheContext = ensureCacheContext();
        final V candidateValue = requireNonNull(supplier.get());
        final byte[] candidateValueBytes = valueMarshaller.marshall(candidateValue);
        final String externalKey = cacheContext.externalEntryKeyFor(internalKey);

        // Loop until either able to add the candidate value, or retrieve one that has been added by another thread
        for (; ; ) {
            final Future<Boolean> addOp = clientSupplier.get().add(externalKey, expiryTime(ttlSeconds), candidateValueBytes);
            if (addOp.get()) {
                // I break here, rather than just return, due to battling with the compiler. Unless written
                // this way, the compiler will not allow the lambda structure.
                break;
            }

            getLogger().info("Cache {}, unable to add candidate for key {}, retrieve what was added", name, internalKey);
            final Optional<V> otherAddedValue = unmarshall((byte[]) clientSupplier.get().get(externalKey), valueMarshaller);
            if (otherAddedValue.isPresent()) {
                cacheContext.recordValue(internalKey, otherAddedValue);
                return otherAddedValue.get();
            }

            getLogger().info("Cache {}, unable to retrieve recently added candidate for key {}, looping", name, internalKey);
        }

        cacheContext.recordValue(internalKey, Optional.of(candidateValue));
        return candidateValue;
    }

    @Nonnull
    @Override
    protected Map<String, V> handleCreation(Function<Set<String>, Map<String, V>> factory, Set<String> externalKeys)
            throws ExecutionException, InterruptedException {
        // Get the missing values from the external cache.
        // Returns map of keys that contain values, so need to handle the missing ones
        final VersionedExternalCacheRequestContext<V> cacheContext = ensureCacheContext();
        final Map<String, Object> haveValues = clientSupplier.get().getBulk(externalKeys);
        getLogger().trace("Cache {}: getBulk(Function): {} of {} entries have values",
                name, haveValues.size(), externalKeys.size());
        final Set<String> missingExternalKeys = new HashSet<>(externalKeys);
        missingExternalKeys.removeAll(haveValues.keySet());

        // Add the retrieved values to the grand result
        final Map<String, V> grandResult = haveValues.entrySet().stream()
                .collect(Collectors.toMap(
                        e -> cacheContext.internalEntryKeyFor(e.getKey()),
                        e -> unmarshall((byte[]) e.getValue(), valueMarshaller).get()));

        if (!missingExternalKeys.isEmpty()) {
            getLogger().trace("Cache {}: getBulk(Function): calling factory to create {} values",
                    name, missingExternalKeys.size());
            // Okay, need to get the missing values and mapping from externalKeys to internalKeys
            final Set<String> missingInternalKeys = Collections.unmodifiableSet(
                    missingExternalKeys.stream().map(cacheContext::internalEntryKeyFor).collect(Collectors.toSet()));
            final Map<String, V> missingValues = factory.apply(missingInternalKeys);
            if (missingInternalKeys.size() != missingValues.size()) {
                getLogger().warn("Cache {}: getBulk(Function): mismatch on generated values, expected ",
                        name, missingInternalKeys.size() + " but got " + missingValues.size());
                throw new ExternalCacheException(ExternalCacheException.Reason.FUNCTION_INCORRECT_RESULT);
            }

            // Okay, got the missing values, now need to add them to Memcached
            final Map<String, Future<Boolean>> internalKeyToFutureMap = missingValues.entrySet().stream().collect(Collectors.toMap(
                    Map.Entry::getKey,
                    e -> clientSupplier.get().set(
                            cacheContext.externalEntryKeyFor(e.getKey()), expiryTime(ttlSeconds), marshall(e.getValue(), valueMarshaller))
            ));

            // Now wait for the outcomes and then add to the grand result
            for (Map.Entry<String, Future<Boolean>> e : internalKeyToFutureMap.entrySet()) {
                e.getValue().get(); // Don't care about the result as it will always be true
            }

            grandResult.putAll(missingValues);
        }

        return grandResult;
    }

    @Nonnull
    @Override
    protected final ExternalCacheException mapException(Exception ex) {
        return MemcachedUtils.mapException(ex);
    }

    @Nonnull
    @Override
    protected final Optional<V> directGet(String externalKey) {
        return unmarshall((byte[]) clientSupplier.get().get(externalKey), valueMarshaller);
    }

    @Nonnull
    @Override
    protected final Map<String, Optional<V>> directGetBulk(Set<String> externalKeys) {
        return MemcachedUtils.directGetBulk(externalKeys, clientSupplier, valueMarshaller);
    }
}
