package com.atlassian.vcache.internal.memcached;

import com.atlassian.marshalling.api.MarshallingPair;
import com.atlassian.vcache.DirectExternalCache;
import com.atlassian.vcache.ExternalCacheSettings;
import com.atlassian.vcache.JvmCache;
import com.atlassian.vcache.JvmCacheSettings;
import com.atlassian.vcache.StableReadExternalCache;
import com.atlassian.vcache.TransactionalExternalCache;
import com.atlassian.vcache.internal.core.service.AbstractVCacheService;
import com.atlassian.vcache.internal.core.service.GuavaJvmCache;
import com.atlassian.vcache.internal.guava.GuavaDirectExternalCache;
import com.atlassian.vcache.internal.guava.GuavaStableReadExternalCache;
import com.atlassian.vcache.internal.guava.GuavaTransactionalExternalCache;
import com.atlassian.vcache.internal.guava.GuavaUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Optional;

import static java.util.Objects.requireNonNull;

/**
 * Main service that is backed by <tt>Memcached</tt>.
 */
public class MemcachedVCacheService extends AbstractVCacheService {
    private static final Logger log = LoggerFactory.getLogger(MemcachedVCacheService.class);

    private final MemcachedVCacheServiceSettings serviceSettings;

    /**
     * Creates an instance using the supplied settings.
     *
     * @param settings the settings to be used for creating the instance.
     * @since 1.3.0
     */
    public MemcachedVCacheService(MemcachedVCacheServiceSettings settings) {
        super(settings.getContextSupplier(),
                settings.getWorkContextContextSupplier(),
                settings.getDefaultsProvider(),
                settings.getCreationHandler(),
                settings.getMetricsCollector(),
                settings.getExternalCacheKeyGenerator(),
                settings.getBegunTransactionalActivityHandler(),
                settings.getLockTimeout());
        this.serviceSettings = requireNonNull(settings);
    }

    @Override
    protected Logger log() {
        return log;
    }

    @Override
    protected <K, V> JvmCache<K, V> createJvmCache(String name, JvmCacheSettings settings) {
        return new GuavaJvmCache<>(name, settings, lockTimeout);
    }

    @Override
    protected <V> TransactionalExternalCache<V> createTransactionalExternalCache(String name,
                                                                                 ExternalCacheSettings settings,
                                                                                 MarshallingPair<V> valueMarshalling,
                                                                                 boolean valueSerializable) {
        if (serviceSettings.getDontExternaliseCache().apply(name)) {
            log.trace("Cache {}: not being externalised", name);
            return new GuavaTransactionalExternalCache<>(
                    name,
                    GuavaUtils.buildDelegate(settings),
                    threadLocalContextSupplier,
                    externalCacheKeyGenerator,
                    (serviceSettings.isSerializationHack() && valueSerializable) ? Optional.empty() : Optional.of(valueMarshalling),
                    transactionControlManager,
                    metricsCollector,
                    lockTimeout);
        }

        return new MemcachedTransactionalExternalCache<>(
                serviceSettings,
                threadLocalContextSupplier,
                externalCacheKeyGenerator,
                name,
                valueMarshalling,
                settings,
                transactionControlManager,
                metricsCollector);
    }

    @Override
    protected <V> StableReadExternalCache<V> createStableReadExternalCache(String name,
                                                                           ExternalCacheSettings settings,
                                                                           MarshallingPair<V> valueMarshalling,
                                                                           boolean valueSerializable) {
        if (serviceSettings.getDontExternaliseCache().apply(name)) {
            log.trace("Cache {}: not being externalised", name);
            return new GuavaStableReadExternalCache<>(
                    name,
                    GuavaUtils.buildDelegate(settings),
                    threadLocalContextSupplier,
                    externalCacheKeyGenerator,
                    (serviceSettings.isSerializationHack() && valueSerializable) ? Optional.empty() : Optional.of(valueMarshalling),
                    metricsCollector,
                    serviceSettings.getLockTimeout());
        }

        return new MemcachedStableReadExternalCache<>(
                serviceSettings,
                threadLocalContextSupplier,
                externalCacheKeyGenerator,
                name,
                valueMarshalling,
                settings,
                metricsCollector);
    }

    @Override
    protected <V> DirectExternalCache<V> createDirectExternalCache(String name,
                                                                   ExternalCacheSettings settings,
                                                                   MarshallingPair<V> valueMarshalling,
                                                                   boolean valueSerializable) {
        if (serviceSettings.getDontExternaliseCache().apply(name)) {
            log.trace("Cache {}: not being externalised", name);
            return new GuavaDirectExternalCache<>(
                    name,
                    GuavaUtils.buildDelegate(settings),
                    threadLocalContextSupplier,
                    externalCacheKeyGenerator,
                    (serviceSettings.isSerializationHack() && valueSerializable) ? Optional.empty() : Optional.of(valueMarshalling),
                    lockTimeout);
        }
        return new MemcachedDirectExternalCache<>(
                serviceSettings,
                threadLocalContextSupplier,
                externalCacheKeyGenerator,
                name,
                valueMarshalling,
                settings);
    }
}
