package com.atlassian.vcache.internal.legacy;

import com.atlassian.cache.Cache;
import com.atlassian.vcache.ExternalCacheException;
import com.atlassian.vcache.Marshaller;
import com.atlassian.vcache.MarshallerException;
import com.atlassian.vcache.PutPolicy;
import com.atlassian.vcache.internal.RequestContext;
import com.atlassian.vcache.internal.core.ExternalCacheKeyGenerator;
import com.atlassian.vcache.internal.core.cas.IdentifiedData;
import com.atlassian.vcache.internal.core.service.AbstractExternalCacheRequestContext;
import com.atlassian.vcache.internal.core.service.AbstractStableReadExternalCache;
import com.atlassian.vcache.internal.core.service.UnversionedExternalCacheRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import static com.atlassian.vcache.internal.core.cas.IdentifiedUtils.marshall;
import static com.atlassian.vcache.internal.core.cas.IdentifiedUtils.unmarshall;
import static java.util.Objects.requireNonNull;

/**
 * Implementation that backs onto Atlassian Cache.
 *
 * @param <V> the value type
 * @since 1.0.0
 */
class LegacyStableReadExternalCache<V>
        extends AbstractStableReadExternalCache<V> {
    private static final Logger log = LoggerFactory.getLogger(LegacyStableReadExternalCache.class);

    private final Cache<String, IdentifiedData> delegate;
    private final Supplier<RequestContext> contextSupplier;
    private final ExternalCacheKeyGenerator keyGenerator;
    private final Optional<Marshaller<V>> valueMarshaller;
    private final LegacyServiceSettings serviceSettings;

    LegacyStableReadExternalCache(
            Cache<String, IdentifiedData> delegate,
            Supplier<RequestContext> contextSupplier,
            ExternalCacheKeyGenerator keyGenerator,
            Optional<Marshaller<V>> valueMarshaller,
            LegacyServiceSettings serviceSettings) {
        super(delegate.getName());
        this.delegate = requireNonNull(delegate);
        this.contextSupplier = requireNonNull(contextSupplier);
        this.keyGenerator = requireNonNull(keyGenerator);
        this.valueMarshaller = requireNonNull(valueMarshaller);
        this.serviceSettings = requireNonNull(serviceSettings);
    }

    @Nonnull
    @Override
    public CompletionStage<Boolean> put(String internalKey, V value, PutPolicy policy) {
        final AbstractExternalCacheRequestContext<V> cacheContext = ensureCacheContext();
        return perform(
                () -> {
                    final String externalKey = cacheContext.externalEntryKeyFor(internalKey);
                    final IdentifiedData identifiedData = marshall(value, valueMarshaller);

                    return LegacyUtils.directPut(externalKey, identifiedData, policy, delegate, serviceSettings.isAvoidCasOps());
                },
                (result) -> {
                    if (result) {
                        cacheContext.recordValue(internalKey, Optional.of(value));
                    } else {
                        cacheContext.forgetValue(internalKey);
                    }
                });
    }

    @Nonnull
    @Override
    public CompletionStage<Void> remove(Iterable<String> internalKeys) {
        return perform(() -> {
            // There is no bulk delete in the api, so need to remove each one
            final AbstractExternalCacheRequestContext<V> cacheContext = ensureCacheContext();
            for (String key : internalKeys) {
                delegate.remove(cacheContext.externalEntryKeyFor(key));
                cacheContext.forgetValue(key);
            }

            return null;
        });
    }

    @Nonnull
    @Override
    public CompletionStage<Void> removeAll() {
        return perform(() -> {
            final AbstractExternalCacheRequestContext<V> cacheContext = ensureCacheContext();
            delegate.removeAll();
            cacheContext.forgetAllValues();
            return null;
        });
    }

    @Nonnull
    @Override
    protected Logger getLogger() {
        return log;
    }

    @Nonnull
    @Override
    protected AbstractExternalCacheRequestContext<V> ensureCacheContext() {
        final RequestContext requestContext = contextSupplier.get();

        return requestContext.computeIfAbsent(this, () -> {
            log.trace("Cache {}: Setting up a new context", delegate.getName());
            return new UnversionedExternalCacheRequestContext<>(
                    keyGenerator, delegate.getName(), requestContext::partitionIdentifier);
        });
    }

    @Nonnull
    @Override
    protected V handleCreation(String internalKey, Supplier<V> supplier)
            throws MarshallerException, ExecutionException, InterruptedException {
        final AbstractExternalCacheRequestContext<V> cacheContext = ensureCacheContext();
        final V candidateValue = requireNonNull(supplier.get());
        final IdentifiedData candidateIdentifiedData = marshall(candidateValue, valueMarshaller);
        final String externalKey = cacheContext.externalEntryKeyFor(internalKey);

        if (serviceSettings.isAvoidCasOps()) {
            delegate.put(externalKey, candidateIdentifiedData);
        } else {
            final Optional<V> otherAddedValue =
                    unmarshall(delegate.putIfAbsent(externalKey, candidateIdentifiedData), valueMarshaller);

            if (otherAddedValue.isPresent()) {
                getLogger().info("Cache {}, unable to add candidate for key {}, use what was added", name, internalKey);
                cacheContext.recordValue(internalKey, otherAddedValue);
                return otherAddedValue.get();
            }
        }

        cacheContext.recordValue(internalKey, Optional.of(candidateValue));
        return candidateValue;
    }

    @Nonnull
    @Override
    protected Map<String, V> handleCreation(Function<Set<String>, Map<String, V>> factory, Set<String> externalKeys)
            throws ExecutionException, InterruptedException {
        // Get the missing values from the external cache.
        // Returns map of keys that contain values, so need to handle the missing ones
        final AbstractExternalCacheRequestContext<V> cacheContext = ensureCacheContext();
        final Map<String, Optional<V>> candidateValues = directGetBulk(externalKeys);

        final Set<String> missingExternalKeys = candidateValues.entrySet().stream()
                .filter(e -> !e.getValue().isPresent())
                .map(Map.Entry::getKey)
                .collect(Collectors.toSet());

        // Add the retrieved values to the grand result
        final Map<String, V> grandResult = candidateValues.entrySet().stream()
                .filter(e -> e.getValue().isPresent())
                .collect(Collectors.toMap(
                        e -> cacheContext.internalEntryKeyFor(e.getKey()),
                        e -> e.getValue().get()));

        if (!missingExternalKeys.isEmpty()) {
            getLogger().trace("Cache {}: getBulk(Function): calling factory to create {} values",
                    name, missingExternalKeys.size());
            // Okay, need to get the missing values and mapping from externalKeys to internalKeys
            final Set<String> missingInternalKeys = Collections.unmodifiableSet(
                    missingExternalKeys.stream().map(cacheContext::internalEntryKeyFor).collect(Collectors.toSet()));
            final Map<String, V> missingValues = factory.apply(missingInternalKeys);
            if (missingInternalKeys.size() != missingValues.size()) {
                getLogger().warn("Cache {}: getBulk(Function): mismatch on generated values, expected ",
                        name, missingInternalKeys.size() + " but got " + missingValues.size());
                throw new ExternalCacheException(ExternalCacheException.Reason.FUNCTION_INCORRECT_RESULT);
            }

            // Okay, got the missing values, now need to add them
            missingValues.entrySet().stream().forEach(e ->
                    delegate.put(
                            cacheContext.externalEntryKeyFor(e.getKey()),
                            marshall(e.getValue(), valueMarshaller)));

            grandResult.putAll(missingValues);
        }

        return grandResult;
    }

    @Nonnull
    @Override
    protected final ExternalCacheException mapException(Exception ex) {
        return LegacyUtils.mapException(ex);
    }

    @Nonnull
    @Override
    protected final Optional<V> directGet(String externalKey) {
        return unmarshall(delegate.get(externalKey), valueMarshaller);
    }

    @Nonnull
    @Override
    protected final Map<String, Optional<V>> directGetBulk(Set<String> externalKeys) {
        return LegacyUtils.directGetBulk(externalKeys, delegate, valueMarshaller);
    }
}
