package com.atlassian.vcache;

import com.atlassian.annotations.PublicApi;

import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * Represents the common operations that can be performed on a JVM local cache.
 *
 * @param <K> the key type
 * @param <V> the value type
 * @since 1.0
 */
@PublicApi
public interface LocalCacheOperations<K, V> {
    /**
     * Returns a value that is associated with a specified key.
     *
     * @param key the key to check.
     * @return an {@link Optional} which may contain the value associated with the key.
     */
    Optional<V> get(K key);

    /**
     * Returns a value that may be associated with a specified key.
     * <p>Notes:</p>
     * <ul>
     * <li>
     * If no entry for the specified key exists, then the specified <tt>supplier</tt> is called to create
     * an entry in the cache, before it is returned.
     * </li>
     * <li>
     * The <tt>supplier</tt> implementation needs to be multi-thread safe as it may be called concurrently
     * if multiple threads call this method.
     * </li>
     * </ul>
     *
     * @param key      the key uniquely identifying the value to be retrieved
     * @param supplier used to generate the value, if one does not exist already for the key. The supplier may not
     *                 return {@code null}.
     * @return the value associated with the key.
     */
    V get(K key, Supplier<? extends V> supplier);

    /**
     * Returns the values that are associated with the specified keys. It is equivalent to calling
     * {@link #getBulk(Function, Iterable)} passing {@code factory} and {@code Arrays.asList(keys)} as the parameters.
     *
     * @param factory used to generate the values for the keys that do not have entries. The factory must return a
     *                map containing only a non-null value for each supplied key. Extra keys are not allowed.
     * @param keys    the keys to retrieve
     * @return A {@link Map} that is keyed on the {@code keys} specified. Each entry in the {@link Map} will have
     * the value associated with the key.
     * @see #getBulk(Function, Iterable)
     */
    @SuppressWarnings("unchecked")
    default Map<K, V> getBulk(Function<Set<K>, Map<K, V>> factory, K... keys) {
        return getBulk(factory, Arrays.asList(keys));
    }

    /**
     * Returns the values that are associated with the specified keys.
     * <p>Notes:</p>
     * <ul>
     * <li>
     * <b>WARNING:</b> this method is not multi-thread safe in conjunction with any methods that make single key
     * mutations, such as {@link #remove(Object)} or {@link #removeIf(Object, Object)}. Caller beware.
     * </li>
     * <li>
     * If there are any keys for which no entry exists, then the given <tt>factory</tt> is called exactly once with
     * the argument being a <tt>Set</tt> containing exactly the keys for which no entry existed.
     * The return value of the <tt>factory</tt> is used to populate the missing entries in the cache,
     * and then the map of all requested keys to values is returned.
     * </li>
     * <li>
     * The call to the factory is made on the thread which calls <tt>getBulk</tt>.
     * </li>
     * <li>
     * Despite the fact that the keys are an <tt>Iterable</tt>, no guarantees are made about the order in which
     * the cache is populated, nor the order in which the <tt>Set</tt> passed to the factory is iterated,
     * nor in which the returned <tt>Map</tt> is iterated.
     * </li>
     * </ul>
     *
     * @param factory used to generate the values for the keys that do not have entries. The factory must return a
     *                map containing only a non-null value for each supplied key. Extra keys are not allowed.
     * @param keys    the keys to retrieve
     * @return A {@link Map} that is keyed on the {@code keys} specified. Each entry in the {@link Map} will have
     * the value associated with the key.
     */
    Map<K, V> getBulk(Function<Set<K>, Map<K, V>> factory, Iterable<K> keys);

    /**
     * Unconditionally puts the value under the specified key.
     *
     * @param key   the key to put the data under
     * @param value the value to associate with the key.
     */
    void put(K key, V value);

    /**
     * Conditionally puts the value under the specified key, if no entry currently exists. Returns the current
     * value associated with the key, if one currently exists.
     *
     * @param key   the key to put the data under
     * @param value the value to associate with the key.
     * @return {@link Optional#EMPTY} is no entry previously existed, otherwise the {@link Optional} contains
     * the current value.
     */
    Optional<V> putIfAbsent(K key, V value);

    /**
     * Conditionally replaces the value associated with a key, iff:
     * <ul>
     * <li>An entry already exists for the key</li>
     * <li>The current value matches the supplied <tt>currentValue</tt> using {@link Object#equals(Object)}</li>
     * </ul>
     *
     * @param key          the key to replace data under
     * @param currentValue the current value to match
     * @param newValue     the replacement value
     * @return whether the current value are replaced with the supplied <tt>newValue</tt>
     */
    boolean replaceIf(K key, V currentValue, V newValue);

    /**
     * Conditionally removed an entry for a specified key, iff:
     * <ul>
     * <li>An entry already exists for the key</li>
     * <li>The current value matches the supplied <tt>value</tt> using {@link Object#equals(Object)}</li>
     * </ul>
     *
     * @param key   the key of the entry to remove conditionally
     * @param value the current value to match
     * @return whether an entry was removed
     */
    boolean removeIf(K key, V value);

    /**
     * Removes the entries for the supplied key.
     *
     * @param key the key of the entry to remove
     */
    void remove(K key);

    /**
     * Removes all the entries in the cache.
     */
    void removeAll();
}
