package com.atlassian.vcache;

import com.atlassian.annotations.PublicApi;

import java.time.Duration;
import java.util.Optional;

/**
 * Builder for creating {@link JvmCacheSettings} instances.
 *
 * @since 1.0
 */
@PublicApi
public class JvmCacheSettingsBuilder {
    private Optional<Integer> maxEntries = Optional.empty();
    private Optional<Duration> defaultTtl = Optional.empty();

    /**
     * Creates an instance with no settings initialised.
     */
    public JvmCacheSettingsBuilder() {
    }

    /**
     * Constructs an instance initialised with the supplied settings.
     *
     * @param settings the settings to configure the instance with.
     */
    public JvmCacheSettingsBuilder(JvmCacheSettings settings) {
        this.maxEntries = settings.getMaxEntries();
        this.defaultTtl = settings.getDefaultTtl();
    }

    /**
     * Returns a new {@link JvmCacheSettings} instance configured using the supplied settings.
     *
     * @return a new {@link JvmCacheSettings} instance configured using the supplied settings.
     */
    public JvmCacheSettings build() {
        return new JvmCacheSettings(maxEntries, defaultTtl);
    }

    /**
     * Specifies the maximum number of entries to be held in the cache. Must not be a negative number.
     * When <tt>0</tt> is specified, the entries will be evicted immediately after being loaded into the cache.
     *
     * @param max the number of entries to be held in the cache
     * @return the builder
     * @throws IllegalArgumentException thrown if <tt>max</tt> is negative
     */
    public JvmCacheSettingsBuilder maxEntries(int max) {
        if (max < 0) {
            throw new IllegalArgumentException("maxEntries must not be negative, passed: " + max);
        }

        maxEntries = Optional.of(max);
        return this;
    }

    /**
     * Specifies the default time-to-live for entries to be held in the cache and must be a positive number.
     * The time-to-live is calculated on the time the entry was added or updated.
     *
     * @param ttl the default time-to-live for entries to be held in the cache
     * @return the builder
     * @throws IllegalArgumentException thrown if <tt>ttl</tt> is not positive
     */
    public JvmCacheSettingsBuilder defaultTtl(Duration ttl) {
        if (ttl.isNegative() || ttl.isZero()) {
            throw new IllegalArgumentException("ttl must be greater than zero, passed: " + ttl);
        }

        defaultTtl = Optional.of(ttl);
        return this;
    }
}
