package com.atlassian.utils.process;

import java.io.InputStream;
import java.io.OutputStream;

/**
 * Interface used to manage the IO and termination needs of an external processes
 * managed by ExternalProcess.  The methods of this interface will be called by
 * different threads, so implementors should take care to ensure thread safety
 *
 * @see com.atlassian.utils.process.ExternalProcess
 */
public interface ProcessHandler {
    /**
     * Process the process stdout stream
     *
     * @param output the external process' output stream (available as an input to this class)
     * @throws ProcessException if there is a problem processing the output
     */
    void processOutput(InputStream output) throws ProcessException;

    /**
     * Process the process stderr stream
     *
     * @param error the external process' standard error stream (available as an input to this class)
     * @throws ProcessException if there is a problem processing the output
     */
    void processError(InputStream error) throws ProcessException;

    /**
     * Indicate if this handler has input to provide to the process
     *
     * @return true if input is available
     */
    boolean hasInput();

    /**
     * Provide input to the external process. Input is provided by writing the content to the
     * given output stream. This method will only be called if hasInput() returns true
     *
     * @param input the output stream representing standard input to the external process
     * @throws IllegalStateException if no input has been configured.
     */
    void provideInput(OutputStream input);

    /**
     * Called when the external process has completed
     *
     * @param exitCode the exit code of the external process
     * @param exception any process exceptions that were thrown within the VM when handling the
     *        external process
     */
    void complete(int exitCode, ProcessException exception);

    /**
     * Indicate if the process has completed
     *
     * @return true if complete has been called.
     */
    boolean isComplete();

    /**
     * Set the watchdog associated with this handler. The watchdog should be called at regular intervals
     * to prevent the external process being terminated. Typically this is done in the IO handling methods
     *
     * @return the handler's watchdog instance
     */
    void setWatchdog(Watchdog watchdog);

    /**
     * Indicate if the process execution has been considered successful.
     *
     * @return true if the process execution completed without error
     */
    boolean succeeded();

    /**
     * Called if the process is to be re-executed.
     */
    void reset();

    /**
     * Get any processing exception associated with this handler
     * @return a processing exception instance or null if no exception occurred.
     */
    ProcessException getException();

    /**
     * Get the process exit code
     * @return process exit code
     */
    int getExitCode();
}

