package com.atlassian.utils.process;

import java.io.Closeable;
import java.io.InputStream;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.UnsupportedEncodingException;

/**
 * An Output Handler which breaks the output from the process into
 * discrete lines and processes each line in turn. Subclasses provide the
 * appropriate implementation of the per-line processing method.
 */
public abstract class LineOutputHandler extends BaseOutputHandler {
    private final String encoding;

    protected LineOutputHandler() {
        this(null); // if not encoding is defined, use the default encoding from the admin
    }

    protected LineOutputHandler(String encoding) {
        this.encoding = encoding;
    }

    public String getEncoding() {
        return encoding;
    }
    
    public void process(InputStream output) throws ProcessException {
        if (encoding == null) {
            process(new BufferReaderLineReader(new BufferedReader(new InputStreamReader(output))));
        } else {
            try {
                process(new BufferReaderLineReader(new BufferedReader(new InputStreamReader(output, encoding))));
            } catch (UnsupportedEncodingException e) {
                throw new ProcessException(e);
            }
        }
    }

    protected void process(LineReadable reader) throws ProcessException {
        int counter = 0;
        try {
            String line;
            while ((line = reader.readLine()) != null) {
                resetWatchdog();
                processLine(counter++, line);
            }
            processInputEnd(counter);
        } catch (InterruptedIOException e) {
            // This means the process was asked to stop which can be normal so we just finish
            processEndByException(counter);
        } catch (IOException e) {
            processEndByException(counter);
            throw new ProcessException(e);
        } finally {
            closeQuietly(reader);
        }
    }

    private void closeQuietly(Closeable closeable) {
        try {
            if (closeable != null) {
                closeable.close();
            }
        } catch (IOException ioe) {
            // ignore
        }
    }

    protected void processEndByException(int counter) {
        // do nothing by default
    }

    /**
     * Input has finished
     *
     * @param lineCount total number of lines processed
     */
    protected void processInputEnd(int lineCount) throws ProcessException {
        // do nothing by default
    }

    /**
     * Process the given line
     *
     * @param lineNum The line number of the line being processed
     * @param line the content of the line
     */
    protected abstract void processLine(int lineNum, String line);
    
    static class BufferReaderLineReader implements LineReadable {

        private final BufferedReader reader;

        BufferReaderLineReader(BufferedReader reader) {
            this.reader = reader;
        } 

        public String readLine() throws IOException {
            return reader.readLine();
        }

        public void close() throws IOException {
            reader.close();
        }
    }

}
