package com.atlassian.user.util;

public class Assert
{
    /**
     * Throws an IllegalArgumentException with a given message when an assertion fails.
     *
     * @param message the message to use in the IllegalArgumentException
     */
    private static void fail(String message)
    {
        throw new IllegalArgumentException(message);
    }

    /**
     * Assert that an object is not <tt>null</tt>, throwing IllegalArgumentException if it is.
     *
     * @param object the object to check for null
     * @param message the exception message to use if the assertion fails
     * @throws IllegalArgumentException if the object is null
     */
    public static <T> T notNull(T object, String message)
    {
        if (object == null)
            fail(message);
        return object;
    }

    /**
     * Asserts that an expression is true, throwing an IllegalArgumentException if it is false.
     *
     * @param expression the expression to check
     * @param message the exception message to use if the assertion fails
     * @throws IllegalArgumentException if the assertion fails
     */
    public static void isTrue(boolean expression, String message)
    {
        if (!expression)
            fail(message);
    }

    /**
     * Assert that the provided object is an instance of the provided class, throwing an IllegalArgumentException if is it not.
     *
     * @param clazz the required class
     * @param object the object to check
     * @throws IllegalArgumentException if the object is not an instance of clazz
     */
    public static <T> T isInstanceOf(Class<T> clazz, Object object)
    {
        if (!clazz.isInstance(object))
            fail(object + " must be an instance of " + clazz);
        return clazz.cast(object);
    }

    /**
     * Assert that the provided object is an instance of the provided class, throwing an IllegalArgumentException if is it not.
     *
     * @param clazz the required class
     * @param object the object to check
     * @param message the exception message to use if the assertion fails
     * @throws IllegalArgumentException if the object is not an instance of clazz
     */
    public static <T> T isInstanceOf(Class<T> clazz, Object object, String message)
    {
        if (!clazz.isInstance(object))
            fail(message);
        return clazz.cast(object);
    }
}
