package com.atlassian.user.search.query;

import com.atlassian.user.*;
import com.atlassian.user.configuration.Configuration;
import com.atlassian.user.configuration.ConfigurationException;
import com.atlassian.user.configuration.util.InitializationCheck;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.DefaultSearchResult;
import com.atlassian.user.search.SearchResult;
import com.atlassian.user.search.query.match.*;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.search.page.PagerUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;

/**
 * This {@link EntityQueryParser} is not efficient, as it handles all queries by loading
 * user and group {@link Entity} objects into memory.
 */
public class DefaultEntityQueryParser implements EntityQueryParser
{
    private static final Logger log = Logger.getLogger(DefaultEntityQueryParser.class);
    private final QueryValidator queryValidator = new QueryValidator();

    protected UserManager userManager;
    protected GroupManager groupManager;
    protected RepositoryIdentifier repository;
    protected Method entityNameMethod;
    protected Method emailMethod;
    protected Method fullnameMethod;
    private static final Class<User> userClass = User.class;

    public DefaultEntityQueryParser(RepositoryIdentifier repo, UserManager userManager, GroupManager groupManager)
    {
        try
        {
            entityNameMethod = userClass.getMethod("getName");
            emailMethod = userClass.getMethod("getEmail");
            fullnameMethod = userClass.getMethod("getFullName");
        }
        catch (NoSuchMethodException e)
        {
            log.error(e.getMessage());
        }

        this.userManager = userManager;
        this.groupManager = groupManager;
        this.repository = repo;
    }

    public void init(HashMap args) throws ConfigurationException
    {
        this.userManager = (UserManager) args.get(Configuration.USERMANAGER);
        this.groupManager = (GroupManager) args.get(Configuration.GROUPMANAGER);
        this.repository = (RepositoryIdentifier) args.get(Configuration.REPOSITORY);

        InitializationCheck.validateArgs(args, new String[]{Configuration.USERMANAGER,
            Configuration.GROUPMANAGER,
            Configuration.REPOSITORY}, this);

        try
        {
            entityNameMethod = userClass.getMethod("getName");
            emailMethod = userClass.getMethod("getEmail");
            fullnameMethod = userClass.getMethod("getFullName");
        }
        catch (NoSuchMethodException e)
        {
            log.error(e.getMessage());
        }
    }

    protected Pager<? extends Entity> parseQuery(Method userMethod, TermQuery q, Pager<? extends Entity> data) throws IllegalAccessException, InvocationTargetException
    {
        String searchTerm = StringUtils.defaultString(q.getTerm()).toLowerCase(); // to allow case insensitive search
        if (searchTerm.indexOf(TermQuery.WILDCARD) >= 0)
            return data;

        Matcher matcher;
        if (q.isMatchingSubstring())
        {
            if (q.getMatchingRule().equals(TermQuery.SUBSTRING_STARTS_WITH))
            {
                matcher = new StartsWithIgnoreCaseMatcher();
            }
            else if (q.getMatchingRule().equals(TermQuery.SUBSTRING_ENDS_WITH))
            {
                matcher = new EndsWithIgnoreCaseMatcher();
            }
            else
            {
                matcher = new ContainsIgnoreCaseMatcher();
            }
        }
        else
        {
            matcher = new EqualsIgnoreCaseMatcher();
        }

        List<Entity> matches = new ArrayList<Entity>();

        for (Entity entity : data)
        {
            String userInfo = (String) userMethod.invoke(entity);

            if (matcher.matches(userInfo, searchTerm))
                matches.add(entity);
        }
        return new DefaultPager<Entity>(matches);
    }

    public Pager find(Query query) throws EntityException
    {
        Pager<? extends Entity> result = null;

        if (query instanceof TermQuery)
        {
            try
            {
                if (query instanceof UserNameTermQuery)
                    result = parseQuery(entityNameMethod, (TermQuery) query, userManager.getUsers());
                else if (query instanceof GroupNameTermQuery)
                    result = parseQuery(entityNameMethod, (TermQuery) query, groupManager.getGroups());
                else if (query instanceof EmailTermQuery)
                    result = parseQuery(emailMethod, (TermQuery) query, userManager.getUsers());
                else if (query instanceof FullNameTermQuery)
                    result = parseQuery(fullnameMethod, (TermQuery) query, userManager.getUsers());
            }
            catch (IllegalAccessException e)
            {
                throw new EntityException(e);
            }
            catch (InvocationTargetException e)
            {
                throw new EntityException(e);
            }
        }
        else if (query instanceof MembershipQuery)
        {
            result = parseMembershipQuery(query);
        }
        else if (query instanceof BooleanQuery)
        {
            BooleanQuery bQuery = (BooleanQuery) query;
            result = evaluateBoolean(bQuery);
        }

        return result;
    }

    private Pager<? extends Entity> parseMembershipQuery(Query query) throws EntityException
    {
        Pager<? extends Entity> result = null;

        if (query instanceof UsersInGroupTwoTermQuery)
        {
            result = parseUsersInGroupTwoTermQuery((UsersInGroupTwoTermQuery) query);
        }
        else if (query instanceof GroupsOfUserTwoTermQuery)
        {
            result = parseGroupsOfUserTwoTermQuery((GroupsOfUserTwoTermQuery) query);
        }

//        TermQuery tQuery = (TermQuery) query;
//
//        if (tQuery.isMatchingSubstring())
//        {
//            GroupNameTermQuery q = new GroupNameTermQuery(tQuery.getTerm(), tQuery.getMatchingRule());
//            Pager pager = find(q);
//            Iterator iter = pager.iterator();
//
//            if (pager.isEmpty())
//                return DefaultPager.emptyPager();
//            else
//            {
//                Pager members = null;
//                while (iter.hasNext())
//                {
//                    Group group = (Group) iter.next();
//
//                    if (members == null)
//                        members = groupManager.getMemberNames(group);
//                    else
//                        members = new MergedPager(members, groupManager.getMemberNames(group));
//                }
//
//                return members;
//            }
//
//        }
//        else
//            return groupManager.getMemberNames(groupManager.getGroup(tQuery.getTerm()));

        return result;
    }

    private Pager parseUsersInGroupTwoTermQuery(UsersInGroupTwoTermQuery query) throws EntityException
    {
        UserNameTermQuery userQuery = query.getUserNameTermQuery();
        GroupNameTermQuery groupQuery = query.getGroupNameTermQuery();

        Pager<User> users = find(userQuery);
        Pager<Group> groups = find(groupQuery);

        Iterator groupsIter = groups.iterator();
        Set<String> candidateUsers = new HashSet<String>();

        while (groupsIter.hasNext())
        {
            Group group = (Group) groupsIter.next();
            candidateUsers.addAll(PagerUtils.toList(groupManager.getLocalMemberNames(group)));
        }

        List userQueryList = PagerUtils.toList(users);

        userQueryList.retainAll(candidateUsers);

        return new DefaultPager(userQueryList);
    }

    private Pager<? extends Entity> evaluateBoolean(BooleanQuery query) throws EntityException
    {
        List queries = query.getQueries();
        Pager<? extends Entity> allResults = null;

        if (query instanceof MembershipQuery)
        {
            return parseMembershipQuery(query);
        }

        boolean anding = query.isAND();

        for (int i = 0; i < queries.size(); i++)
        {
            Query nextQuery = (Query) queries.get(i);
            List initialResult;

            try
            {
                if (allResults == null)
                {
                    allResults = find(nextQuery);
                }
                else if (nextQuery instanceof BooleanQuery)
                {
                    if (anding)
                    {
                        Pager<? extends Entity> resultsToAnd = evaluateBoolean((BooleanQuery) nextQuery);
                        List<? extends Entity> allResultsList = PagerUtils.toList(allResults);

                        List<Entity> resultsToAndList = new ArrayList<Entity>(PagerUtils.toList(resultsToAnd));
                        resultsToAndList.retainAll(allResultsList);
                        allResults = new DefaultPager<Entity>(resultsToAndList);
                    }
                    else
                    {
                        Pager<? extends Entity> resultsToOr = evaluateBoolean((BooleanQuery) nextQuery);
                        List<Entity> resultsToOrList = new ArrayList<Entity>(PagerUtils.toList(resultsToOr));
                        List<Entity> intersection = findIntersection(PagerUtils.toList(allResults), resultsToOrList);
                        allResults = new DefaultPager<Entity>(intersection);
                    }
                }
                else if (anding)
                {

                    if (nextQuery instanceof UserNameTermQuery)
                    {
                        initialResult = PagerUtils.toList(parseQuery(entityNameMethod, (TermQuery) nextQuery,
                            allResults));
                        initialResult.addAll(PagerUtils.toList(allResults));
                        allResults = new DefaultPager(initialResult);
                    }
                    else if (nextQuery instanceof GroupNameTermQuery)
                    {
                        initialResult = PagerUtils.toList(parseQuery(entityNameMethod, (TermQuery) nextQuery,
                            allResults));
                        initialResult.addAll(PagerUtils.toList(allResults));
                        allResults = new DefaultPager(initialResult);
                    }
                    else if (nextQuery instanceof EmailTermQuery)
                    {
                        initialResult = PagerUtils.toList(parseQuery(emailMethod, (TermQuery) nextQuery, allResults));
                        initialResult.addAll(PagerUtils.toList(allResults));
                        allResults = new DefaultPager(initialResult);
                    }
                    else if (nextQuery instanceof FullNameTermQuery)
                    {
                        initialResult = PagerUtils.toList(parseQuery(fullnameMethod, (TermQuery) nextQuery,
                            allResults));
                        initialResult.addAll(PagerUtils.toList(allResults));
                        allResults = new DefaultPager(initialResult);
                    }
                }
                else // OR
                {
                    initialResult = PagerUtils.toList(find(nextQuery));
                    List intersection = findIntersection(PagerUtils.toList(allResults), initialResult);
                    allResults = new DefaultPager(intersection);
                }
            }
            catch (Exception e)
            {
                log.error(e.getClass().getName() + " - " + e.getMessage());
            }
        }

        return allResults;
    }

    private <T> List<T> findIntersection(List<? extends T> list1, List<? extends T> list2)
    {
        List<T> result = new ArrayList<T>(list1);

        list2.removeAll(list1);
        result.addAll(list2);

        return result;
    }

    private Pager parseGroupsOfUserTwoTermQuery(GroupsOfUserTwoTermQuery query) throws EntityException
    {
        UserNameTermQuery userQuery = query.getUserNameTermQuery();
        GroupNameTermQuery groupQuery = query.getGroupNameTermQuery();

        Pager groups = find(groupQuery);
        Pager users = find(userQuery);

        Iterator usersIter = users.iterator();
        Set<Group> candidateGroups = new HashSet<Group>();

        while (usersIter.hasNext())
        {
            User user = (User) usersIter.next();
            candidateGroups.addAll(PagerUtils.toList(groupManager.getGroups(user)));
        }

        List groupQueryList = PagerUtils.toList(groups);

        groupQueryList.retainAll(candidateGroups);

        return new DefaultPager(groupQueryList);
    }

    public SearchResult findUsers(Query query) throws EntityException
    {
        queryValidator.assertValid(query);
        Pager pager = find(query);
        return new DefaultSearchResult(pager, repository.getKey());
    }

    public SearchResult findGroups(Query query) throws EntityException
    {
        queryValidator.assertValid(query);
        Pager pager = find(query);
        return new DefaultSearchResult(pager, repository.getKey());
    }

    public SearchResult findUsers(Query query, QueryContext context) throws EntityException
    {
        if (!context.contains(repository))
            return null;

        queryValidator.assertValid(query);
        return findUsers(query);
    }

    public SearchResult findGroups(Query query, QueryContext context) throws EntityException
    {
        if (!context.contains(repository))
            return null;

        queryValidator.assertValid(query);
        return findGroups(query);
    }
}