package com.atlassian.user.impl.hibernate.search.query;

import com.atlassian.user.Entity;
import com.atlassian.user.EntityException;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.hibernate.DefaultHibernateExternalEntity;
import com.atlassian.user.impl.hibernate.DefaultHibernateGroup;
import com.atlassian.user.impl.hibernate.DefaultHibernateUser;
import com.atlassian.user.impl.hibernate.repository.HibernateRepository;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.DefaultSearchResult;
import com.atlassian.user.search.SearchResult;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.query.*;
import net.sf.hibernate.Criteria;
import net.sf.hibernate.HibernateException;
import net.sf.hibernate.Session;
import net.sf.hibernate.expression.*;
import org.springframework.orm.hibernate.SessionFactoryUtils;

import java.util.*;

/**
 * Handles {@link Query} objects on a {@link HibernateRepository}.
 */
public final class HibernateEntityQueryParser implements EntityQueryParser
{
    private final RepositoryIdentifier identifier;
    private final HibernateRepository repository;
    private final QueryValidator queryValidator = new QueryValidator();

    public HibernateEntityQueryParser(RepositoryIdentifier identifier, HibernateRepository repository)
    {
        this.identifier = identifier;
        this.repository = repository;
    }

    public SearchResult findUsers(Query query) throws EntityException
    {
        queryValidator.assertValid(query);
        return parseQuery(query);
    }

    public SearchResult findGroups(Query query) throws EntityException
    {
        queryValidator.assertValid(query);
        return parseQuery(query);
    }

    public SearchResult findUsers(Query query, QueryContext context) throws EntityException
    {
        if (!context.contains(identifier))
            return null;

        queryValidator.assertValid(query);
        return parseQuery(query);
    }

    public SearchResult findGroups(Query query, QueryContext context) throws EntityException
    {
        if (!context.contains(identifier))
            return null;

        queryValidator.assertValid(query);
        return parseQuery(query);
    }

    private MatchMode getMatchMode(String matchingRule)
    {
        if (matchingRule.equals(TermQuery.SUBSTRING_CONTAINS))
        {
            return MatchMode.ANYWHERE;
        }
        if (matchingRule.equals(TermQuery.SUBSTRING_ENDS_WITH))
        {
            return MatchMode.END;
        }
        if (matchingRule.equals(TermQuery.SUBSTRING_STARTS_WITH))
        {
            return MatchMode.START;
        }
        return MatchMode.EXACT;
    }

    private String identifyProperty(TermQuery q)
    {
        if (q instanceof UserNameTermQuery)
            return "name";
        else if (q instanceof EmailTermQuery)
            return "email";
        else if (q instanceof FullNameTermQuery)
            return "fullName";
        else if (q instanceof GroupNameTermQuery)
            return "name";
        else if (q instanceof GroupsOfUserTwoTermQuery)
            return "entity";
        return null;
    }

    /**
     * Converts a {@link BooleanQuery} object into an appropriate Hibernate {@link Criteria}, performs the search, and
     * returns a {@link SearchResult}.
     */
    private SearchResult parseQuery(Query query) throws EntityException
    {
        //1. establish the Object being queried.
        Session session = SessionFactoryUtils.getSession(repository.getSessionFactory(), true);
        Query definingQuery = (query instanceof BooleanQuery) ? identifyDefiningQuery((BooleanQuery) query) : query;

        List result;
        try
        {
            Criteria baseCriteria = getBaseCriteria(definingQuery, session);
            baseCriteria = identifyAndAddSearchCriteria(query, definingQuery, baseCriteria);
            baseCriteria.addOrder(Order.asc("name")); // sort results explicitly by username rather than leaving this up to the mood of the database

            result = baseCriteria.list();
        }
        catch (HibernateException e)
        {
            throw new RepositoryException(e);
        }

        return new DefaultSearchResult(new DefaultPager(result), identifier.getName());
    }

    private Criteria identifyAndAddSearchCriteria(Query q, Query definingQuery, Criteria baseCriteria)
        throws EntityQueryException, HibernateException
    {
        if (q instanceof BooleanQuery)
            return addSearchCriteria((BooleanQuery) q, definingQuery, baseCriteria);
        else
        {
            return addSearchCriteria((TermQuery) q, baseCriteria);
        }
    }

    private Criteria addSearchCriteria(BooleanQuery q, Query definingQuery, Criteria baseCriteria)
        throws EntityQueryException, HibernateException
    {
        /**
         * Membership queries are performed on associations - i.e. subCriteria, and need to be handled on a case by case basis.
         */
        if (definingQuery instanceof MembershipQuery)
        {
            return addMembershipSearchCriteria(q, baseCriteria);
        }

        Junction junction = identifyAndOrJunction(q);
        baseCriteria.add(junction);

        for (Query query : q.getQueries())
        {
            if (query instanceof BooleanQuery)
            {
                addSearchCriteria((BooleanQuery) query, definingQuery, baseCriteria);
            }
            else if (query instanceof TermQuery)
            {
                junction.add(getQueryExpression((TermQuery) query));
            }
            else
                throw new EntityQueryException("Unknown query type: [" + query.getClass().getName() + "]");
        }

        return baseCriteria;
    }

    private Junction identifyAndOrJunction(BooleanQuery q)
    {
        Junction junction;
        if (q.isAND())
            junction = Expression.conjunction();
        else
            junction = Expression.disjunction();

        return junction;
    }

    private Criteria addMembershipSearchCriteria(BooleanQuery q, Criteria baseCriteria) throws HibernateException
    {
        if (q instanceof GroupsOfUserTwoTermQuery)
        {
            addGroupsOfUserSearchCriteria(q, baseCriteria);
        }
        else if (q instanceof GroupsOfExternalEntityTwoTermQuery)
        {
            addGroupsOfExternalEntitySearchCriteria(q, baseCriteria);
        }

        return baseCriteria;
    }

    private void addGroupsOfUserSearchCriteria(BooleanQuery q, Criteria baseCriteria) throws HibernateException
    {
        UserNameTermQuery userNameQuery = ((GroupsOfUserTwoTermQuery) q).getUserNameTermQuery();
        GroupNameTermQuery groupNameQuery = ((GroupsOfUserTwoTermQuery) q).getGroupNameTermQuery();

        if (groupNameQuery.getTerm().equals(TermQuery.WILDCARD))
        {
            //do nothing, there is no specific group query and we defer to the username term below
        }
        else if (groupNameQuery.isMatchingSubstring())
        {
            baseCriteria.add(getLikeExpression("name", groupNameQuery, false));
        }
        else
        {
            baseCriteria.add(new EqExpression("name", groupNameQuery.getTerm(), false));
        }

        if (userNameQuery.isMatchingSubstring())
        {
            baseCriteria.createCriteria("localMembers").add(getLikeExpression("name", userNameQuery, false));
        }
        else
        {
            baseCriteria.createCriteria("localMembers").add(new EqExpression("name", userNameQuery.getTerm(), false));
        }
    }

    private Criterion getLikeExpression(String entityAttribute, TermQuery termQuery, boolean caseInsensitive)
    {
        if (caseInsensitive)
            return Expression.ilike(entityAttribute, termQuery.getTerm(), getMatchMode(termQuery.getMatchingRule()));
        else
            return Expression.like(entityAttribute, termQuery.getTerm(), getMatchMode(termQuery.getMatchingRule()));
    }

    private void addGroupsOfExternalEntitySearchCriteria(BooleanQuery q, Criteria baseCriteria)
        throws HibernateException
    {
        ExternalEntityNameTermQuery nameQuery =
            ((GroupsOfExternalEntityTwoTermQuery) q).getExternalEntityNameTermQuery();
        GroupNameTermQuery groupQuery = ((GroupsOfExternalEntityTwoTermQuery) q).getGroupNameTermQuery();

        if (groupQuery.getTerm().equals(TermQuery.WILDCARD))
        {
            //do nothing, there is no specific group query and we defer to the username term below
        }
        else if (groupQuery.isMatchingSubstring())
        {
            baseCriteria.add(getLikeExpression("name", groupQuery, false));
        }
        else
        {
            baseCriteria.add(new EqExpression("name", groupQuery.getTerm(), false));
        }

        if (nameQuery.isMatchingSubstring())
        {
            baseCriteria.createCriteria("externalMembers").add(getLikeExpression("name", nameQuery, false));
        }
        else
        {
            baseCriteria.createCriteria("externalMembers").add(new EqExpression("name", nameQuery.getTerm(), false));
        }
    }

    private Criteria addSearchCriteria(TermQuery q, Criteria baseCriteria)
    {
        Criterion expression = getQueryExpression(q);
        baseCriteria.add(expression);
        return baseCriteria;
    }

    private Criterion getQueryExpression(TermQuery termQuery)
    {
        String hqlField = identifyProperty(termQuery);

        if (termQuery.isMatchingSubstring())
        {
            return getLikeExpression(hqlField, termQuery, true);
        }
        else
        {
            return new EqExpression(hqlField, termQuery.getTerm(), true);
        }
    }

    private static final Map<Class<? extends Query>, Class<? extends Entity>> QUERY_TYPE_CLASSES = new LinkedHashMap<Class<? extends Query>, Class<? extends Entity>>();

    static {
        QUERY_TYPE_CLASSES.put(UserQuery.class, DefaultHibernateUser.class);
        QUERY_TYPE_CLASSES.put(GroupQuery.class, DefaultHibernateGroup.class);
        QUERY_TYPE_CLASSES.put(UsersInGroupTwoTermQuery.class, DefaultHibernateUser.class);
        QUERY_TYPE_CLASSES.put(GroupsOfUserTwoTermQuery.class, DefaultHibernateGroup.class);
        QUERY_TYPE_CLASSES.put(GroupsOfExternalEntityTwoTermQuery.class, DefaultHibernateGroup.class);
        QUERY_TYPE_CLASSES.put(ExternalEntitiesInGroupTwoTermQuery.class, DefaultHibernateExternalEntity.class);
    }

    private Criteria getBaseCriteria(Query query, Session session)
    {
        for (Map.Entry<Class<? extends Query>, Class<? extends Entity>> entry : QUERY_TYPE_CLASSES.entrySet())
        {
            if (entry.getKey().isInstance(query))
            {
                return session.createCriteria(entry.getValue());
            }
        }
        return null;
    }

    /**
     * The 'defining query' lets us know which EntityClass we are querying.
     */
    private Query identifyDefiningQuery(BooleanQuery q)
    {
        if (q instanceof MembershipQuery)
            return q;
        for (Query query : q.getQueries())
        {
            if (query instanceof TermQuery)
                return query;
            if (query instanceof BooleanQuery)
                return identifyDefiningQuery((BooleanQuery) query);
        }
        return null;
    }
}
