package com.atlassian.user.impl.cache;

import com.atlassian.user.User;
import com.atlassian.user.Group;
import com.atlassian.cache.*;

import java.util.List;
import java.util.Iterator;

/**
 * The membership check cache is a cache where the key is the {@link String}
 * returned by {@link #getMembershipKey(String, Group)} for a given user
 * and group, and the value is a {@link Boolean} indicating whether the user
 * is a member of that group.
 * <p/>
 * Both true and false membership checks are cached.
 */
public class MembershipCache
{
    private final CacheFactory cacheFactory;
    private final String cacheName;

    public MembershipCache(CacheFactory cacheFactory, String cacheName)
    {
        this.cacheFactory = cacheFactory;
        this.cacheName = cacheName;
    }

    private com.atlassian.cache.Cache getCache()
    {
        return cacheFactory.getCache(cacheName);
    }

    protected String getMembershipKey(String username, Group group)
    {
        return username + "_" + group.getName();
    }

    public void put(User user, Group group, boolean isMember)
    {
        getCache().put(getMembershipKey(user.getName(), group), Boolean.valueOf(isMember));
    }

    /**
     * @return {@link Boolean#TRUE} if the user is a member of the group,
     *         {@link Boolean#FALSE} if the user is not a member of the group, or
     *         null if the result is not in the cache.
     */
    public Boolean get(User user, Group group)
    {
        return (Boolean) getCache().get(getMembershipKey(user.getName(), group));
    }

    public void remove(User user, Group group)
    {
        getCache().remove(getMembershipKey(user.getName(), group));
    }

    /**
     * Remove all cached values for a list of users and an associated group.
     * Typically called to remove all membership information when a group is
     * removed.
     *
     * @param usernames a {@link List} of username {@link String}s.
     * @param group the group of which the users are members.
     */
    public void remove(List usernames, Group group)
    {
        for (Iterator iter = usernames.iterator(); iter.hasNext();)
        {
            String username = (String) iter.next();
            getCache().remove(getMembershipKey(username, group));
        }
    }
}
