package com.atlassian.user.configuration;

import com.atlassian.user.GroupManager;
import com.atlassian.user.UserManager;
import com.atlassian.user.properties.PropertySetFactory;
import com.atlassian.user.util.ClassLoaderUtils;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.beans.factory.support.RootBeanDefinition;

import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

public class DefaultRepositoryProcessor implements RepositoryProcessor
{
    protected final Logger log = Logger.getLogger(this.getClass());

    public RepositoryAccessor process(RepositoryConfiguration config) throws ConfigurationException
    {
        DefaultListableBeanFactory beanFactory = createComponentContext(config);

        beanFactory.registerSingleton("identifier", config.getIdentifier());

        List componentNames = Arrays.asList(new String[]{
            Configuration.PROPERTYSET_FACTORY,
            Configuration.PASSWORD_ENCRYPTOR,
            Configuration.USERMANAGER,
            Configuration.AUTHENTICATOR,
            Configuration.GROUPMANAGER,
            Configuration.ENTITY_QUERY_PARSER,
        });
        for (Iterator it = componentNames.iterator(); it.hasNext();)
        {
            String component = (String) it.next();
            if (!config.hasClassForComponent(component)) continue;
            beanFactory.registerBeanDefinition(component,
                createBeanDefinition(config.getComponentClassName(component), true));
        }

        beanFactory.registerBeanDefinition(Configuration.CLASS,
            createBeanDefinition(config.getComponentClassName(Configuration.CLASS), false));
        DefaultRepositoryAccessor accessor = (DefaultRepositoryAccessor) beanFactory.getBean(Configuration.CLASS);

        if (config.isCachingEnabled())
        {
            CacheConfiguration cacheConfiguration = config.getCacheConfiguration();
            beanFactory.registerBeanDefinition("cachingUserManager", createBeanDefinition(
                cacheConfiguration.getUserManagerClassName(), true));
            beanFactory.registerBeanDefinition("cachingGroupManager", createBeanDefinition(
                cacheConfiguration.getGroupManagerClassName(), true));
            beanFactory.registerBeanDefinition("cachingPropertySetFactory", createBeanDefinition(
                cacheConfiguration.getPropertySetFactoryClassName(), true));

            accessor.setUserManager((UserManager) beanFactory.getBean("cachingUserManager"));
            accessor.setGroupManager((GroupManager) beanFactory.getBean("cachingGroupManager"));
            
            if (beanFactory.containsBeanDefinition("propertySetFactory"))
                accessor.setPropertySetFactory((PropertySetFactory) beanFactory.getBean("cachingPropertySetFactory"));
        }

        return accessor;
    }

    protected Object createBean(String componentName, RepositoryConfiguration config)
    {
        return createBean(componentName, config, true);
    }

    protected Object createBean(String componentName, RepositoryConfiguration config,
        boolean useConstructorInjection)
    {
        DefaultListableBeanFactory beanFactory = createComponentContext(config);
        String className = config.getComponentClassName(componentName);
        if(className == null)
        {
            throw new RuntimeException("expected component [ "+componentName+" ] does not specify a className");
        }
        beanFactory.registerBeanDefinition(componentName, createBeanDefinition(className, useConstructorInjection));
        return beanFactory.getBean(componentName);
    }

    private RootBeanDefinition createBeanDefinition(String className, boolean useConstructorInjection)
    {
        int autowireMode = useConstructorInjection ? RootBeanDefinition.AUTOWIRE_CONSTRUCTOR :
            RootBeanDefinition.AUTOWIRE_BY_TYPE;
        return new RootBeanDefinition(getClassForName(className), autowireMode);
    }

    /**
     * Gets the specified class using {@link ClassLoaderUtils#loadClass(String,Class)}.
     *
     * @param className the fully qualified name of the class to retrieve
     * @throws RuntimeException if the class cannot be found.
     */
    protected Class getClassForName(String className)
    {
        try
        {
            return ClassLoaderUtils.loadClass(className, this.getClass());
        }
        catch (ClassNotFoundException e)
        {
            throw new RuntimeException("Could not find class: [" + className + "]", e);
        }
    }

    /**
     * Create a Spring bean factory containing the components in the provided map.
     *
     * @return the bean factory containing the components.
     */
    protected DefaultListableBeanFactory createComponentContext(RepositoryConfiguration config)
    {
        DefaultListableBeanFactory beanFactory = new DefaultListableBeanFactory();
        for (Iterator iter = config.getComponentNames().iterator(); iter.hasNext();)
        {
            String name = (String) iter.next();
            Object component = config.getComponent(name);
            beanFactory.registerSingleton(name, component);
        }
        return beanFactory;
    }
}
