package com.atlassian.upm.osgi.impl;

import com.atlassian.upm.osgi.Version;

import static java.lang.String.format;
import static org.apache.commons.lang.StringUtils.isEmpty;
import static org.apache.commons.lang.StringUtils.isNotEmpty;
import static org.apache.commons.lang.StringUtils.isNumeric;

/**
 * A wrapper class around OSGi versions
 */
public final class VersionImpl implements Version
{
    private final int major;
    private final int minor;
    private final int micro;
    private final String qualifier;

    VersionImpl(org.osgi.framework.Version version)
    {
        this.major = version.getMajor();
        this.minor = version.getMinor();
        this.micro = version.getMicro();
        this.qualifier = version.getQualifier();
    }

    public int getMajor()
    {
        return major;
    }

    public int getMinor()
    {
        return minor;
    }

    public int getMicro()
    {
        return micro;
    }

    public String getQualifier()
    {
        return qualifier;
    }

    public int compareTo(Version version)
    {
        if (this == version)
        {
            return 0;
        }

        int result = getMajor() - version.getMajor();
        if (result != 0)
        {
            return result;
        }

        result = getMinor() - version.getMinor();
        if (result != 0)
        {
            return result;
        }

        result = getMicro() - version.getMicro();
        if (result != 0)
        {
            return result;
        }

        /**
         * In some cases, such as milestone releases, non-blank qualifiers signify versions prior to a
         * {@link Version} with a blank qualifier. For example, 2.1.0-m5 and 2.1.0.
         *
         * In other cases, such as 2.1.0.1 and 2.1.0, the qualifier may signify a bug fix version.
         *
         * We will consider all numeric qualifiers to be greater than blank qualifiers,
         * and all non-numeric qualifiers to be less than blank qualifiers.
         * Non-blank qualifiers will have standard String comparisons against each other.
         */
        String otherQualifier = version.getQualifier();
        if (isEmpty(getQualifier()) && isNotEmpty(otherQualifier))
        {
            return compareQualifierWithBlankQualifier(otherQualifier, false);
        }
        else if (isNotEmpty(getQualifier()) && isEmpty(otherQualifier))
        {
            return compareQualifierWithBlankQualifier(getQualifier(), true);
        }
        else
        {
            return getQualifier().compareTo(version.getQualifier());
        }
    }

    private static int compareQualifierWithBlankQualifier(String qualifier, boolean thisQualifier)
    {
        if (isNumeric(qualifier))
        {
            return "".compareTo(qualifier) * (thisQualifier ? -1 : 1);
        }
        else
        {
            return qualifier.compareTo("") * (thisQualifier ? -1 : 1);
        }
    }

    public boolean equals(Object other)
    {
        if (this == other)
        {
            return true;
        }

        if (other == null || !(other instanceof Version))
        {
            return false;
        }

        return compareTo((Version) other) == 0;
    }

    public int hashCode()
    {
        return 31 * (31 * (31 * major + minor) + micro) + qualifier.hashCode();
    }

    public String toString()
    {
        String result = format("%d.%d.%d", major, minor, micro);
        if (qualifier.length() != 0)
        {
            result += "." + qualifier;
        }
        return result;
    }
}
