package com.atlassian.upm.license.role.jira;

import java.net.URI;

import com.atlassian.jira.permission.GlobalPermissionKey;
import com.atlassian.jira.permission.GlobalPermissionType;
import com.atlassian.jira.security.GlobalPermissionManager;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.user.util.UserManager;
import com.atlassian.upm.license.role.spi.AbstractLicensingRole;

import static com.atlassian.fugue.Option.option;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * JIRA implementation of licensing roles. Currently this is based on pluggable global permissions.
 */
public class JiraLicensingRole extends AbstractLicensingRole
{
    private final GlobalPermissionManager permissionManager;
    private final UserManager userManager;
    private final GlobalPermissionKey permission;
    private final GlobalPermissionUserCountCache globalPermissionUserCountCache;

    public JiraLicensingRole(GlobalPermissionManager permissionManager,
                             UserManager userManager,
                             GlobalPermissionType permission,
                             GlobalPermissionUserCountCache globalPermissionUserCountCache)
    {
        super(permission.getKey(), permission.getNameI18nKey(), permission.getDescriptionI18nKey());
        this.permissionManager = checkNotNull(permissionManager, "permissionManager");
        this.userManager = checkNotNull(userManager, "userManager");
        this.permission = checkNotNull(permission.getGlobalPermissionKey(), "permission");
        this.globalPermissionUserCountCache = checkNotNull(globalPermissionUserCountCache, "globalPermissionUserCountCache");
    }

    @Override
    public URI getManagementPage()
    {
        // Service Desk wanted a custom role management page. We decided against offering this to third-party vendors
        // as we felt that a consistent experience would lead to a better user experience, and that efforts should
        // be funneled instead towards improving the underlying host application management pages.
        if ("com.atlassian.servicedesk.agent.access".equals(getKey()))
        {
            return URI.create("/secure/admin/AgentAllocation.jspa");
        }

        return URI.create("/secure/admin/GlobalPermissions!default.jspa");
    }

    @Override
    public boolean isUserInRole(String userKey)
    {
        for (ApplicationUser user : option(userManager.getUserByKey(userKey)))
        {
            return permissionManager.hasPermission(permission, user);
        }

        return false;
    }

    @Override
    public int getRoleCount()
    {
        return globalPermissionUserCountCache.getUserCount(permission);
    }
}
