package com.atlassian.upm.license.role.jira;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import com.atlassian.cache.Cache;
import com.atlassian.cache.CacheLoader;
import com.atlassian.cache.CacheManager;
import com.atlassian.cache.CacheSettingsBuilder;
import com.atlassian.crowd.embedded.api.CrowdService;
import com.atlassian.crowd.embedded.api.User;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.atlassian.jira.permission.GlobalPermissionKey;
import com.atlassian.jira.permission.GlobalPermissionType;
import com.atlassian.jira.security.GlobalPermissionManager;
import com.atlassian.jira.user.ApplicationUsers;
import com.atlassian.util.concurrent.NotNull;

public class GlobalPermissionUserCountCacheImpl implements GlobalPermissionUserCountCache
{
    private final GlobalPermissionManager globalPermissionManager;
    private final Cache<GlobalPermissionKey, Integer> permissionCountCache;

    public GlobalPermissionUserCountCacheImpl(GlobalPermissionManager globalPermissionManager,
                                              CrowdService crowdService,
                                              CacheManager cacheManager)
    {
        this.globalPermissionManager = globalPermissionManager;
        permissionCountCache = cacheManager.getCache("com.atlassian.upm.license.role.jira.GlobalPermissionUserCountCache",
                new GlobalPermissionUserCountCacheLoader(crowdService),
                new CacheSettingsBuilder().expireAfterAccess(1, TimeUnit.HOURS).build());
    }

    @Override
    public void resetAll()
    {
        permissionCountCache.removeAll();
    }

    @Override
    public void reset(GlobalPermissionKey globalPermissionKey)
    {
        permissionCountCache.remove(globalPermissionKey);
    }

    @Override
    public int getUserCount(GlobalPermissionKey globalPermissionKey)
    {
        return permissionCountCache.get(globalPermissionKey);
    }

    private class GlobalPermissionUserCountCacheLoader implements CacheLoader<GlobalPermissionKey, Integer>
    {
        private final CrowdService crowdService;

        public GlobalPermissionUserCountCacheLoader(CrowdService crowdService)
        {
            this.crowdService = crowdService;
        }

        private Iterable<User> getGroupMembers(final String groupName)
        {
            final MembershipQuery<User> membershipQuery =
                    QueryBuilder.queryFor(User.class, EntityDescriptor.user())
                                .childrenOf(EntityDescriptor.group())
                                .withName(groupName)
                                .returningAtMost(EntityQuery.ALL_RESULTS);
            return crowdService.search(membershipQuery);
        }

        @Override
        public Integer load(@NotNull final GlobalPermissionKey permissionKey)
        {
            final Collection<String> groupsWithUsePermission = globalPermissionManager.getGroupNamesWithPermission(permissionKey);
            final Set<User> allUsers = new HashSet<User>();

            for (final String groupName : groupsWithUsePermission)
            {
                Iterable<User> users = getGroupMembers(groupName);
                for (User user : users)
                {
                    if (user.isActive() && hasUsePermission(user))
                    {
                        allUsers.add(user);
                    }
                }
            }
            return allUsers.size();
        }

        private boolean hasUsePermission(User user)
        {
            /**
             * Users can "use JIRA" as long as they have any of the following permissions.
             */
            for (String permission : GlobalPermissionType.getUsePermissions())
            {
                if (globalPermissionManager.hasPermission(GlobalPermissionKey.of(permission), ApplicationUsers.from(user)))
                {
                    return true;
                }
            }

            return false;
        }
    }
}
