package com.atlassian.upm.license.role.confluence;

import com.atlassian.fugue.Option;
import com.atlassian.upm.license.role.spi.LicensingRoleCreationFailedException;
import com.atlassian.user.EntityException;
import com.atlassian.user.Group;
import com.atlassian.user.GroupManager;
import com.atlassian.user.User;
import com.atlassian.user.UserManager;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.atlassian.fugue.Option.none;
import static com.atlassian.fugue.Option.option;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Iterables.size;

/**
 * A facade around some Atlassian User manager implementations which like to throw exceptions.
 */
public class ConfluenceRoleBasedLicenseManager
{
    private static final Logger log = LoggerFactory.getLogger(ConfluenceRoleBasedLicenseManager.class);

    private final GroupManager groupManager;
    private final UserManager userManager;

    public ConfluenceRoleBasedLicenseManager(GroupManager groupManager, UserManager userManager)
    {
        this.groupManager = checkNotNull(groupManager, "groupManager");
        this.userManager = checkNotNull(userManager, "userManager");
    }

    protected Group createGroup(String groupName) throws LicensingRoleCreationFailedException
    {
        try
        {
            return groupManager.createGroup(groupName);
        }
        catch (EntityException e)
        {
            throw new LicensingRoleCreationFailedException(e);
        }
    }

    protected Option<User> getUser(String userKey)
    {
        try
        {
            return option(userManager.getUser(userKey));
        }
        catch (EntityException e)
        {
            logException("Failed to look up user with userKey: " + userKey, e);
        }

        return none();
    }

    protected Option<Group> getGroup(final String groupName)
    {
        try
        {
            return option(groupManager.getGroup(groupName));
        }
        catch (EntityException e)
        {
            logException("Failed to look up group with name: " + groupName, e);
        }

        return none();
    }

    protected boolean hasMembership(Group group, String userKey)
    {
        try
        {
            for (User user : getUser(userKey))
            {
                return groupManager.hasMembership(group, user);
            }
        }
        catch (EntityException e)
        {
            logException("Failed to look up user with userKey: " + userKey, e);
        }

        return false;
    }

    protected int getMembershipCount(String groupName)
    {
        try
        {
            // Important: prior to ever using this in a production environment, this will need to be
            // re-implemented in a more performant fashion.
            return size(groupManager.getMemberNames(groupManager.getGroup(groupName)));
        }
        catch (EntityException e)
        {
            logException("Failed to look up group membership count with name: " + groupName, e);
        }

        return 0;
    }

    private void logException(String errorMessage, Exception e)
    {
        log.warn(errorMessage);
        log.debug(errorMessage, e);
    }
}
