package com.atlassian.upm.license.storage.lib;

import java.net.URI;
import java.util.concurrent.Callable;

import com.atlassian.plugin.PluginAccessor;
import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.upm.SysCommon;
import com.atlassian.upm.api.license.PluginLicenseManager;
import com.atlassian.upm.api.license.entity.PluginLicense;
import com.atlassian.upm.api.util.Option;
import com.atlassian.upm.license.storage.plugin.PluginLicenseStorageManager;

import com.google.common.base.Function;

import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import static com.atlassian.upm.license.storage.lib.VersionChecker.isUpm201OrLaterInstalled;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.lang.Boolean.parseBoolean;
import static java.lang.System.getProperty;

public class ThirdPartyPluginLicenseStorageManagerImpl implements ApplicationContextAware, ThirdPartyPluginLicenseStorageManager
{
    private ManagerAccessor managerAccessor;
    private final ApplicationProperties applicationProperties;
    private final PluginAccessor pluginAccessor;
    private final UserManager userManager;

    public ThirdPartyPluginLicenseStorageManagerImpl(ApplicationProperties applicationProperties,
                                                     PluginAccessor pluginAccessor,
                                                     UserManager userManager)
    {
        this(applicationProperties, pluginAccessor, userManager, null);
    }

    /**
     * Package-scoped constructor to allow mocking our {@link ManagerAccessor}.
     */
    public ThirdPartyPluginLicenseStorageManagerImpl(ApplicationProperties applicationProperties,
                                                     PluginAccessor pluginAccessor,
                                                     UserManager userManager,
                                                     ManagerAccessor managerAccessor)
    {
        this.applicationProperties = checkNotNull(applicationProperties, "applicationProperties");
        this.pluginAccessor = checkNotNull(pluginAccessor, "pluginAccessor");
        this.userManager = checkNotNull(userManager, "userManager");
        this.managerAccessor = managerAccessor;
    }

    @Override
    public Option<PluginLicense> getLicense() throws PluginLicenseStoragePluginUnresolvedException
    {
        Callable<Option<PluginLicense>> withUpm2 = new Callable<Option<PluginLicense>>()
        {
            @Override
            public Option<PluginLicense> call() throws Exception
            {
                return getLicenseManager().getLicense();
            }
        };

        Callable<Option<PluginLicense>> withoutUpm2 = new Callable<Option<PluginLicense>>()
        {
            @Override
            public Option<PluginLicense> call() throws Exception
            {
                return getStorageManager().getLicense();
            }
        };

        return execute(withUpm2, withoutUpm2, false);
    }

    @Override
    public Option<String> getRawLicense() throws PluginLicenseStoragePluginUnresolvedException
    {
        Callable<Option<String>> withUpm2 = new Callable<Option<String>>()
        {
            @Override
            public Option<String> call() throws Exception
            {
                return getLicenseManager().getLicense().map(new Function<PluginLicense, String>()
                {
                    @Override
                    public String apply(PluginLicense license)
                    {
                        return license.getRawLicense();
                    }
                });
            }
        };

        Callable<Option<String>> withoutUpm2 = new Callable<Option<String>>()
        {
            @Override
            public Option<String> call() throws Exception
            {
                return getStorageManager().getRawLicense();
            }
        };

        return execute(withUpm2, withoutUpm2, false);
    }

    @Override
    public Option<String> setRawLicense(final String rawLicense) throws PluginLicenseStoragePluginUnresolvedException
    {
        Callable<Option<String>> withUpm2 = new Callable<Option<String>>()
        {
            @Override
            public Option<String> call() throws Exception
            {
                throw new UnsupportedOperationException("Cannot set license manually when UPM is license-aware. " +
                        "Please use UPM's licensing UI.");
            }
        };

        Callable<Option<String>> withoutUpm2 = new Callable<Option<String>>()
        {
            @Override
            public Option<String> call() throws Exception
            {
                return getStorageManager().setRawLicense(rawLicense);
            }
        };

        return execute(withUpm2, withoutUpm2, true);
    }

    @Override
    public Option<String> removeRawLicense() throws PluginLicenseStoragePluginUnresolvedException
    {
        Callable<Option<String>> withUpm2 = new Callable<Option<String>>()
        {
            @Override
            public Option<String> call() throws Exception
            {
                throw new UnsupportedOperationException("Cannot remove license manually when UPM is license-aware. " +
                        "Please use UPM's licensing UI.");
            }
        };

        Callable<Option<String>> withoutUpm2 = new Callable<Option<String>>()
        {
            @Override
            public Option<String> call() throws Exception
            {
                return getStorageManager().removeRawLicense();
            }
        };

        return execute(withUpm2, withoutUpm2, true);
    }

    @Override
    public Option<PluginLicense> validateLicense(final String rawLicense) throws PluginLicenseStoragePluginUnresolvedException
    {
        Callable<Option<PluginLicense>> withUpm2 = new Callable<Option<PluginLicense>>()
        {
            @Override
            public Option<PluginLicense> call() throws Exception
            {
                throw new UnsupportedOperationException("Cannot validate license manually when UPM is license-aware. " +
                        "Please use UPM's licensing UI.");
            }
        };

        Callable<Option<PluginLicense>> withoutUpm2 = new Callable<Option<PluginLicense>>()
        {
            @Override
            public Option<PluginLicense> call() throws Exception
            {
                return getStorageManager().validateLicense(rawLicense);
            }
        };

        return execute(withUpm2, withoutUpm2, false);
    }

    @Override
    public boolean isUpmLicensingAware()
    {
        return isUpm201OrLaterInstalled(pluginAccessor);
    }

    @Override
    public String getPluginKey() throws PluginLicenseStoragePluginUnresolvedException
    {
        Callable<String> withUpm2 = new Callable<String>()
        {
            @Override
            public String call() throws Exception
            {
                return getLicenseManager().getPluginKey();
            }
        };

        Callable<String> withoutUpm2 = new Callable<String>()
        {
            @Override
            public String call() throws Exception
            {
                return getStorageManager().getPluginKey();
            }
        };

        return execute(withUpm2, withoutUpm2, false);
    }

    @Override
    public URI getPluginManagementUri() throws PluginLicenseStoragePluginUnresolvedException
    {
        return URI.create(applicationProperties.getBaseUrl()
                + "/plugins/servlet/upm?fragment=manage/"
                + getPluginKey()).normalize();
    }

    @Override
    public boolean isOnDemand()
    {
        return SysCommon.isOnDemand();
    }


    /**
     * Executes either the "with UPM 2" or "without UPM 2" functionality depending on whether
     * or not a licensing aware UPM is present.
     *
     * If a {@link ClassCastException} is thrown from the "with UPM 2" functionality, this likely comes from a
     * plugin being wired to the wrong {@link PluginLicenseManager} class instance. As a result, the "without UPM 2"
     * version will then be attempted.
     *
     * @param withUpm2 functionality to execute if a licensing-aware UPM is present
     * @param withoutUpm2 functionality to execute if a licensing-aware UPM is not present
     * @param enforceAdmin true if administrator privileges should be enforced, false if all users should be allowed
     * @param <T> the type to return
     * @return the return value of the executed {@link Callable}
     * @throws PluginLicenseStoragePluginUnresolvedException
     *
     */
    private <T> T execute(Callable<T> withUpm2, Callable<T> withoutUpm2, boolean enforceAdmin) throws PluginLicenseStoragePluginUnresolvedException
    {
        if (enforceAdmin && !hasAdminPermission())
        {
            throw new PluginLicenseStoragePluginPermissionDeniedException(userManager.getRemoteUsername());
        }

        // UPM-2005 register all OSGi wires before executing any licensing code
        registerOsgiWires();

        try
        {
            if (isUpmLicensingAware())
            {
                try
                {
                    return withUpm2.call();
                }
                // UPM-1932 Certain plugin configurations break our Plugin License Storage plugin installation process.
                // To work around this they need to deploy their plugin as an OBR and remove their dynamic import package statements.
                // However, removing these statements will cause ClassCastExceptions when a customer first updates from
                // UPM 1.x to 2.x because the plugin will not re-wire to the newly-found version of PluginLicenseManager.
                //
                // This isn't the cleanest of solutions, but it should work around this problem.
                // All other functionality will continue working as is.
                catch (ClassCastException e)
                {
                    return withoutUpm2.call();
                }
            }
            else
            {
                return withoutUpm2.call();
            }
        }
        //re-throw the exception that was thrown rather than wrapping it in another exception
        catch (UnsupportedOperationException e)
        {
            throw e;
        }
        //re-throw the exception that was thrown rather than wrapping it in another exception
        catch (IllegalArgumentException e)
        {
            throw e;
        }
        //re-throw the exception that was thrown rather than wrapping it in another exception
        catch (PluginLicenseStoragePluginUnresolvedException e)
        {
            throw e;
        }
        //an unexpected exception occurred from the nested Callable
        catch (Exception e)
        {
            throw new PluginLicenseStoragePluginUnresolvedException(e);
        }
    }

    private PluginLicenseStorageManager getStorageManager() throws PluginLicenseStoragePluginUnresolvedException
    {
        return managerAccessor.getStorageManager();
    }

    private PluginLicenseManager getLicenseManager()
    {
        return managerAccessor.getLicenseManager();
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException
    {
        this.managerAccessor = new ManagerAccessor(applicationContext);
    }

    /**
     * Checks whether or not the current user is an administrator. Prior to 2.2.1, this was only checked
     * in the generated license administration code and not in this component.
     *
     * @since 2.2.1
     * @return true if the current user has admin privileges, false otherwise
     */
    private boolean hasAdminPermission()
    {
        String user = userManager.getRemoteUsername();
        try
        {
            return user != null && (userManager.isAdmin(user) || userManager.isSystemAdmin(user));
        }
        catch(NoSuchMethodError e)
        {
            // userManager.isAdmin(String) was not added until SAL 2.1.
            // We need this check to ensure backwards compatibility with older product versions.
            return user != null && userManager.isSystemAdmin(user);
        }
    }

    /**
     * The following is what happens *without* this method's workaround.
     *
     * When the Plugin License Storage plugin (PLSP) is used for a Marketplace plugin's licensing solution
     * (for example when UPM 2 is absent), only two wires are created between the Marketplace plugin and PLSP.
     * The Marketplace plugin looks like the following in Felix:
     *
     *     com.atlassian.upm.api.util,version=2.2.2 from com.atlassian.upm.plugin-license-storage-plugin (93)
     *     com.atlassian.upm.license.storage.plugin,version=2.2.2 from com.atlassian.upm.plugin-license-storage-plugin (93)
     *
     * The packages com.atlassian.upm.api.license and com.atlassian.upm.api.license.entity are not wired
     * to anything at this time, even though they are specified as dynamic import packages.
     * 
     * At some later point, customers will install a licensing-aware UPM of a version newer than this PLSP.
     * The Marketplace plugin will use that UPM's licensing API instead of PLSP's. At this time the
     * Marketplace plugin will wire the remaining packages to UPM, however, the existing wires are not rewired
     * (even with dynamic import packages) until the entire bundle is brought back down and back up. As a
     * result, the Marketplace plugin now looks like the following in Felix:
     *
     *     com.atlassian.upm.api.license,version=2.2.3 from com.atlassian.upm.atlassian-universal-plugin-manager-plugin (94)
     *     com.atlassian.upm.api.util,version=2.2.2 from com.atlassian.upm.plugin-license-storage-plugin (93)
     *     com.atlassian.upm.license.storage.plugin,version=2.2.2 from com.atlassian.upm.plugin-license-storage-plugin (93)
     *
     * This is a problem because now classes in com.atlassian.upm.license (such as PluginLicenseManager) and
     * classes in the Marketplace plugin have different versions of other API classes such as Option. See UPM-2005 for
     * additional information.
     *
     * To work around this, we can force all wires to be created at a single time, such that they will all
     * wire to the same bundle. With this fix, the Marketplace plugin will look like the following in Felix,
     * both before and after installing a newer UPM (and until the next product restart):
     *
     *     com.atlassian.upm.api.license,version=2.2.2 from com.atlassian.upm.plugin-license-storage-plugin (93)
     *     com.atlassian.upm.api.license.entity,version=2.2.2 from com.atlassian.upm.plugin-license-storage-plugin (93)
     *     com.atlassian.upm.api.util,version=2.2.2 from com.atlassian.upm.plugin-license-storage-plugin (93)
     *     com.atlassian.upm.license.storage.plugin,version=2.2.2 from com.atlassian.upm.plugin-license-storage-plugin (93)
     *
     * @since 2.2.4
     * 
     */
    @SuppressWarnings("unused")
    private void registerOsgiWires()
    {
        // Trigger com.atlassian.upm.api.util wire
        Class<Option> clazz1 = Option.class;
        // Trigger com.atlassian.upm.api.license.entity wire
        Class<PluginLicense> clazz2 = PluginLicense.class;
        // Trigger com.atlassian.upm.api.license wire
        Class<PluginLicenseManager> clazz3 = PluginLicenseManager.class;
    }
}
