package com.atlassian.upm.license.storage.lib;

import com.atlassian.upm.UpmVersion;
import com.atlassian.upm.api.license.HostLicenseInformation;
import com.atlassian.upm.api.license.PluginLicenseManager;
import com.atlassian.upm.license.storage.plugin.PluginLicenseStorageManager;

import org.springframework.beans.factory.config.AutowireCapableBeanFactory;
import org.springframework.context.ApplicationContext;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Package-scoped class to dynamically load {@link PluginLicenseStorageManager} and/or
 * {@link PluginLicenseManager}, if either are available.
 */
class ManagerAccessor
{
    private final LazyReference<ThirdPartyPluginLicenseStorageManagerAccessor> thirdPartyAccessor;
    private final LazyReference<PluginLicenseManagerAccessor> licenseManagerAccessor;
    private final LazyReference<UpmLicenseInformationAccessor> upmLicenseInformationAccessor;

    ManagerAccessor(final ApplicationContext applicationContext)
    {
        thirdPartyAccessor = new LazyReference<ThirdPartyPluginLicenseStorageManagerAccessor>()
        {
            @Override
            protected ThirdPartyPluginLicenseStorageManagerAccessor create() throws Exception
            {
                try
                {
                    Class<?> accessorImplClass = getClass().getClassLoader().loadClass(
                            "com.atlassian.upm.license.storage.lib.ThirdPartyPluginLicenseStorageManagerAccessorImpl");
                    return (ThirdPartyPluginLicenseStorageManagerAccessor) applicationContext.getAutowireCapableBeanFactory().
                            createBean(accessorImplClass, AutowireCapableBeanFactory.AUTOWIRE_CONSTRUCTOR, false);
                }
                catch (Exception e)
                {
                    throw new PluginLicenseStoragePluginUnresolvedException(e);
                }
            }
        };

        licenseManagerAccessor = new LazyReference<PluginLicenseManagerAccessor>()
        {
            @Override
            protected PluginLicenseManagerAccessor create() throws Exception
            {
                try
                {
                    Class<?> accessorImplClass = getClass().getClassLoader().loadClass(
                            "com.atlassian.upm.license.storage.lib.PluginLicenseManagerAccessorImpl");
                    return (PluginLicenseManagerAccessor) applicationContext.getAutowireCapableBeanFactory().
                            createBean(accessorImplClass, AutowireCapableBeanFactory.AUTOWIRE_CONSTRUCTOR, false);
                }
                catch (Exception e)
                {
                    throw new RuntimeException("Unable to find plugin license manager", e);
                }
            }
        };

        upmLicenseInformationAccessor = new LazyReference<UpmLicenseInformationAccessor>()
        {
            @Override
            protected UpmLicenseInformationAccessor create() throws Exception
            {
                try
                {
                    Class<?> accessorImplClass = getClass().getClassLoader().loadClass(
                        "com.atlassian.upm.license.storage.lib.UpmLicenseInformationAccessorImpl");
                    return (UpmLicenseInformationAccessor) applicationContext.getAutowireCapableBeanFactory().
                        createBean(accessorImplClass, AutowireCapableBeanFactory.AUTOWIRE_CONSTRUCTOR, false);
                }
                catch (Exception e)
                {
                    throw new RuntimeException("Unable to find UPM License Information", e);
                }
            }
        };
    }

    PluginLicenseStorageManager getStorageManager() throws PluginLicenseStoragePluginUnresolvedException
    {
        return thirdPartyAccessor.get().getPluginLicenseStorageManager();
    }

    PluginLicenseManager getLicenseManager()
    {
        return licenseManagerAccessor.get().getPluginLicenseManager();
    }

    HostLicenseInformation getUpmHostLicenseInformation()
    {
        return upmLicenseInformationAccessor.get().getHostLicenseInformation();
    }
}
