package com.atlassian.marketplace.client.model;

import java.net.URI;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import com.atlassian.marketplace.client.api.ApplicationKey;
import com.atlassian.upm.api.util.Option;
import com.atlassian.upm.api.util.Pair;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;

import static com.atlassian.upm.api.util.Option.none;
import static com.atlassian.upm.api.util.Option.some;
import static com.atlassian.upm.api.util.Pair.pair;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Classes for constructing model objects.  Normally all model objects will be returned from API
 * methods, but in some cases (in test code or if you are making changes to the returned objects)
 * you may wish to construct them yourselves.  Do not use the constructors of the model classes
 * directly,  since they may change depending on implementation details of the JSON encoding scheme.
 * <p>
 * Most model objects have some required properties.  These builders provide default values for all
 * such properties, to facilitate their use in test code; be sure to change the values of all properties
 * you care about.  The default values are guaranteed to be valid in the API schema.  All optional
 * properties default to {@link Option#none()}; collections default to an empty list.
 */
public abstract class ModelBuilders
{
    public static String DEFAULT_STRING = "DefaultValue";
    public static Date DEFAULT_DATE = new Date(0L);
    public static int DEFAULT_INT = Integer.MAX_VALUE;
    public static long DEFAULT_LONG = Long.MAX_VALUE;
    public static URI DEFAULT_URI = URI.create("http://default/uri");
    public static int DEFAULT_STARS = 3;
    public static float DEFAULT_PRICE = 100.00f;
    
    public static ApplicationBuilder application()
    {
        return new ApplicationBuilder();
    }
    
    public static ApplicationBuilder application(Application from)
    {
        return application().links(from.getLinks()).key(from.getKey()).name(from.getName())
            .order(from.getOrder()).pluginCount(from.getPluginCount()).versions(from.getVersions())
            .categories(from.getCategories());
    }

    public static ApplicationSummaryBuilder applicationSummary()
    {
        return new ApplicationSummaryBuilder();
    }
    
    public static ApplicationSummaryBuilder applicationSummary(ApplicationSummary from)
    {
        return applicationSummary().links(from.getLinks()).key(from.getKey()).name(from.getName())
            .order(from.getOrder()).pluginCount(from.getPluginCount());
    }
    
    public static ApplicationSummaryBuilder applicationSummary(Application from)
    {
        return applicationSummary().links(from.getLinks()).key(from.getKey()).name(from.getName())
            .order(from.getOrder()).pluginCount(from.getPluginCount());
    }
    
    public static ApplicationVersionBuilder applicationVersion()
    {
        return new ApplicationVersionBuilder();
    }
    
    public static ApplicationVersionBuilder applicationVersion(ApplicationVersion from)
    {
        return applicationVersion().links(from.getLinks()).buildNumber(from.getBuildNumber())
            .published(from.isPublished()).releaseDate(from.getReleaseDate()).version(from.getVersion());
    }
    
    public static BannerBuilder banner()
    {
        return new BannerBuilder();
    }
    
    
    public static IconsBuilder icons()
    {
        return new IconsBuilder();
    }
    
    public static IconsBuilder icons(Icons from)
    {
        return icons().defaultIcon(from.getDefaultIcon()).tinyIcon(from.getTinyIcon());
    }
    
    public static ImageBuilder image()
    {
        return new ImageBuilder();
    }
    
    public static ImageBuilder image(Image from)
    {
        return image().links(from.getLinks()).altText(from.getAltText()).height(from.getHeight())
            .width(from.getWidth());
    }
    
    public static LicenseTypeBuilder licenseType()
    {
        return new LicenseTypeBuilder();
    }
    
    public static LicenseTypeBuilder licenseType(LicenseType from)
    {
        return licenseType().links(from.getLinks()).id(from.getId()).name(from.getName());
    }
    
    public static LinksBuilder links()
    {
        return new LinksBuilder();
    }

    public static LinksBuilder links(Links links)
    {
        return links().put(links);
    }

    public static ModificationBuilder modification()
    {
        return new ModificationBuilder();
    }
    
    public static ModificationBuilder modification(Modification from)
    {
        return modification().by(from.getBy()).date(from.getDate());
    }
    
    public static PluginBuilder plugin()
    {
        return new PluginBuilder();
    }
    
    public static PluginBuilder plugin(Plugin from)
    {
        return plugin().links(from.getLinks()).categories(from.getCategories())
            .compatibleApplications(from.getCompatibleApplications()).creationDate(from.getCreationDate())
            .deployable(from.isDeployable()).description(from.getDescription()).downloadCount(from.getDownloadCount())
            .lastModified(from.getLastModified()).media(from.getMedia()).name(from.getName())
            .oldVersion(from.isOldVersion()).pluginKey(from.getPluginKey()).pricing(from.getPricing())
            .reviews(from.getReviews()).reviewSummary(from.getReviewSummary()).summary(from.getSummary())
            .vendor(from.getVendor()).version(from.getVersion()).versions(from.getVersions())
            .versionsCount(from.getVersionCount());
    }

    public static PluginCategoryBuilder pluginCategory()
    {
        return new PluginCategoryBuilder();
    }
    
    public static PluginCategoryBuilder pluginCategory(PluginCategory from)
    {
        return pluginCategory().links(from.getLinks()).name(from.getName());
    }
    
    public static PluginMediaBuilder pluginMedia()
    {
        return new PluginMediaBuilder();
    }
    
    public static PluginMediaBuilder pluginMedia(PluginMedia from)
    {
        return pluginMedia().banner(from.getBanner()).icons(from.getIcons()).screenshots(from.getScreenshots())
            .youtubeId(from.getYoutubeId());
    }
    
    public static PluginSummaryBuilder pluginSummary()
    {
        return new PluginSummaryBuilder();
    }
    
    public static PluginSummaryBuilder pluginSummary(PluginSummary from)
    {
        return pluginSummary().links(from.getLinks()).categories(from.getCategories())
            .compatibleApplications(from.getCompatibleApplications()).downloadCount(from.getDownloadCount())
            .lastModified(from.getLastModified()).lastReleasedDate(from.getLastReleasedDate())
            .icons(from.getIcons()).name(from.getName()).pluginKey(from.getPluginKey()).pricing(from.getPricing())
            .reviewSummary(from.getReviewSummary()).summary(from.getSummary())
            .vendor(from.getVendor()).version(from.getVersion());
    }

    public static PluginSummaryBuilder pluginSummary(Plugin from)
    {
        return pluginSummary().links(from.getLinks()).categories(from.getCategories())
            .compatibleApplications(from.getCompatibleApplications()).downloadCount(from.getDownloadCount())
            .lastModified(from.getLastModified()).icons(from.getMedia().getIcons())
            .name(from.getName()).pluginKey(from.getPluginKey()).pricing(from.getPricing())
            .reviewSummary(from.getReviewSummary()).summary(from.getSummary())
            .vendor(from.getVendor()).version(from.getVersion());
    }
    
    public static PluginVersionBuilder pluginVersion()
    {
        return new PluginVersionBuilder();
    }
    
    public static PluginVersionBuilder pluginVersion(PluginVersion from)
    {
        return pluginVersion().links(from.getLinks()).compatibilities(from.getCompatibilities())
            .buildNumber(from.getBuildNumber())
            .compatibleApplications(from.getCompatibleApplications()).deployable(from.isDeployable())
            .license(from.getLicense()).marketplaceType(from.getMarketplaceType())
            .marketplaceTypeTooltip(from.getMarketplaceTypeTooltip())
            .pluginSystemVersion(from.getPluginSystemVersion())
            .published(from.isPublished())
            .releaseDate(from.getReleaseDate())
            .releasedBy(from.getReleasedBy()).releaseFocus(from.getReleaseFocus())
            .releaseNotes(from.getReleaseNotes()).screenshots(from.getScreenshots()).stable(from.isStable())
            .summary(from.getSummary()).supportType(from.getSupportType())
            .version(from.getVersion());
    }
    
    public static PricingBuilder pricing()
    {
        return new PricingBuilder();
    }
    
    public static PricingBuilder pricing(Pricing from)
    {
        return pricing().links(from.getLinks()).expertDiscountOptOut(from.isExpertDiscountOptOut())
            .items(from.getItems()).lastModified(from.getLastModified()).legacyItems(from.getLegacyItems())
            .legacyMaxVersion(from.getLegacyMaxVersion()).nonLegacyStartVersion(from.getNonLegacyStartVersion())
            .parent(from.getParent());
    }
    
    public static PricingItemBuilder pricingItem()
    {
        return new PricingItemBuilder();
    }
    
    public static PricingItemBuilder pricingItem(PricingItem from)
    {
        return pricingItem().description(from.getDescription()).enterprise(from.isEnterprise())
            .id(from.getId()).legacyDisplay(from.isLegacyDisplay()).licenseType(from.getLicenseType())
            .saleType(from.getSaleType()).starter(from.isStarter()).unitCount(from.getUnitCount())
            .unitDescription(from.getUnitDescription()).usdAmount(from.getUsdAmount());
    }
    
    public static ReviewBuilder review()
    {
        return new ReviewBuilder();
    }
    
    public static ReviewBuilder review(Review from)
    {
        return review().author(from.getAuthor()).date(from.getDate()).pluginName(from.getPluginName())
            .response(from.getResponse()).review(from.getReview()).stars(from.getStars());
    }
    
    public static ReviewResponseBuilder reviewResponse()
    {
        return new ReviewResponseBuilder();
    }
    
    public static ReviewResponseBuilder reviewResponse(ReviewResponse from)
    {
        return reviewResponse().response(from.getResponse());
    }
    
    public static ReviewsBuilder reviews()
    {
        return new ReviewsBuilder();
    }
    
    public static ReviewsBuilder reviews(Reviews from)
    {
        return reviews().links(from.getLinks()).reviews(from.getReviews());
    }
    
    public static ReviewSummaryBuilder reviewSummary()
    {
        return new ReviewSummaryBuilder();
    }
    
    public static ReviewSummaryBuilder reviewSummary(ReviewSummary from)
    {
        return reviewSummary().links(from.getLinks()).averageStars(from.getAverageStars())
            .count(from.getCount());
    }
    
    public static ScreenshotBuilder screenshot()
    {
        return new ScreenshotBuilder();
    }
    
    public static ScreenshotBuilder screenshot(Screenshot from)
    {
        return screenshot().carouselImage(from.getCarouselImage()).image(from.getImage()).name(from.getName());
    }
    
    public static UserSummaryBuilder userSummary()
    {
        return new UserSummaryBuilder();
    }
    
    public static UserSummaryBuilder userSummary(UserSummary from)
    {
        return userSummary().links(from.getLinks()).email(from.getEmail()).id(from.getId()).name(from.getName());
    }
    
    public static VendorBuilder vendor()
    {
        return new VendorBuilder();
    }
    
    public static VendorBuilder vendor(Vendor from)
    {
        return vendor().links(from.getLinks()).email(from.getEmail()).id(from.getId()).name(from.getName())
            .otherContactDetails(from.getOtherContactDetails()).phone(from.getPhone());
    }
    
    public static VersionCompatibilityBuilder versionCompatibility()
    {
        return new VersionCompatibilityBuilder();
    }
    
    public static VersionCompatibilityBuilder versionCompatibility(VersionCompatibility from)
    {
        return versionCompatibility().applicationName(from.getApplicationName()).max(from.getMax()).min(from.getMin());
    }
    
    public static abstract class BuilderWithLinks<T extends BuilderWithLinks<T>>
    {
        protected LinksBuilder links = new LinksBuilder();
     
        @SuppressWarnings("unchecked")
        public T links(Links links)
        {
            this.links.removeAll();
            this.links.put(links);
            return (T) this;
        }
        
        @SuppressWarnings("unchecked")
        public T addLink(String rel, URI uri)
        {
            this.links.put(rel, uri);
            return (T) this;
        }
        
        @SuppressWarnings("unchecked")
        public T addLink(String rel, String type, URI uri)
        {
            this.links.put(rel, some(type), uri);
            return (T) this;
        }
    }
    
    public static final class ApplicationBuilder extends BuilderWithLinks<ApplicationBuilder>
    {
        private ApplicationKey key = ApplicationKey.valueOf(DEFAULT_STRING);
        private String name = DEFAULT_STRING;
        private int order = DEFAULT_INT;
        private int pluginCount = DEFAULT_INT;
        private Collection<ApplicationVersion> versions = ImmutableList.of();
        private Collection<String> categories = ImmutableList.of();
     
        public Application build()
        {
            return new Application(links.build(), key.getKey(), name, order, pluginCount, versions, categories);
        }
        
        public ApplicationBuilder key(ApplicationKey key)
        {
            this.key = checkNotNull(key);
            return this;
        }

        public ApplicationBuilder name(String name)
        {
            this.name = checkNotNull(name);
            return this;
        }

        public ApplicationBuilder order(int order)
        {
            this.order = order;
            return this;
        }

        public ApplicationBuilder pluginCount(int pluginCount)
        {
            this.pluginCount = pluginCount;
            return this;
        }
        
        public ApplicationBuilder versions(Iterable<ApplicationVersion> versions)
        {
            this.versions = ImmutableList.copyOf(checkNotNull(versions));
            return this;
        }
        
        public ApplicationBuilder categories(Iterable<String> categories)
        {
            this.categories = ImmutableList.copyOf(checkNotNull(categories));
            return this;
        }
    }

    public static final class ApplicationSummaryBuilder extends BuilderWithLinks<ApplicationSummaryBuilder>
    {
        private ApplicationKey key = ApplicationKey.valueOf(DEFAULT_STRING);
        private String name = DEFAULT_STRING;
        private int order = DEFAULT_INT;
        private int pluginCount = DEFAULT_INT;
     
        public ApplicationSummary build()
        {
            return new ApplicationSummary(links.build(), key.getKey(), name, order, pluginCount);
        }
        
        public ApplicationSummaryBuilder key(ApplicationKey key)
        {
            this.key = checkNotNull(key);
            return this;
        }

        public ApplicationSummaryBuilder name(String name)
        {
            this.name = checkNotNull(name);
            return this;
        }

        public ApplicationSummaryBuilder order(int order)
        {
            this.order = order;
            return this;
        }

        public ApplicationSummaryBuilder pluginCount(int pluginCount)
        {
            this.pluginCount = pluginCount;
            return this;
        }
    }

    public static final class ApplicationVersionBuilder extends BuilderWithLinks<ApplicationVersionBuilder>
    {
        private long buildNumber = DEFAULT_LONG;
        private String version = DEFAULT_STRING;
        private Date releaseDate = DEFAULT_DATE;
        private boolean published = false;
        
        public ApplicationVersion build()
        {
            String status = published ? "Published" : "Draft";
            return new ApplicationVersion(links.build(), buildNumber, version, releaseDate,
                                          new KeyValuePair(status, status));
        }
        
        public ApplicationVersionBuilder buildNumber(long buildNumber)
        {
            this.buildNumber = buildNumber;
            return this;
        }

        public ApplicationVersionBuilder version(String version)
        {
            this.version = checkNotNull(version);
            return this;
        }

        public ApplicationVersionBuilder releaseDate(Date releaseDate)
        {
            this.releaseDate = releaseDate;
            return this;
        }
        
        public ApplicationVersionBuilder published(boolean published)
        {
            this.published = published;
            return this;
        }
    }
    
    public static final class BannerBuilder
    {
        private String pluginKey = DEFAULT_STRING;
        private Image image = ModelBuilders.image().build();
        private Option<Image> carouselImage = none();
        
        public Banner build()
        {
            return new Banner(pluginKey, image, carouselImage.getOrElse((Image) null));
        }
        
        public BannerBuilder pluginKey(String pluginKey)
        {
            this.pluginKey = checkNotNull(pluginKey);
            return this;
        }
        
        public BannerBuilder image(Image image)
        {
            this.image = checkNotNull(image);
            return this;
        }

        public BannerBuilder carouselImage(Option<Image> carouselImage)
        {
            this.carouselImage = checkNotNull(carouselImage);
            return this;
        }
    }
    
    public static final class IconsBuilder
    {
        private Option<Image> defaultIcon = none();
        private Option<Image> tinyIcon = none();

        public Icons build()
        {
            return new Icons(defaultIcon.getOrElse((Image) null), tinyIcon.getOrElse((Image) null));
        }
        
        public IconsBuilder defaultIcon(Option<Image> defaultIcon)
        {
            this.defaultIcon = checkNotNull(defaultIcon);
            return this;
        }
        
        public IconsBuilder tinyIcon(Option<Image> tinyIcon)
        {
            this.tinyIcon = checkNotNull(tinyIcon);
            return this;
        }
    }
    
    public static final class ImageBuilder extends BuilderWithLinks<ImageBuilder>
    {
        private int width = DEFAULT_INT;
        private int height = DEFAULT_INT;
        private Option<String> altText = none();
        
        public ImageBuilder()
        {
            links.put("binary", DEFAULT_URI);
        }
        
        public Image build()
        {
            return new Image(links.build(), width, height, nullableString(altText));
        }

        public ImageBuilder width(int width)
        {
            this.width = width;
            return this;
        }

        public ImageBuilder height(int height)
        {
            this.height = height;
            return this;
        }

        public ImageBuilder altText(Option<String> altText)
        {
            this.altText = checkNotNull(altText);
            return this;
        }
    }
    
    public static class LicenseTypeBuilder extends BuilderWithLinks<LicenseTypeBuilder>
    {
        private String id = DEFAULT_STRING;
        private String name = DEFAULT_STRING;
        
        public LicenseTypeBuilder()
        {
            links.put("alternate", DEFAULT_URI);
        }
        
        public LicenseType build()
        {
            return new LicenseType(links.build(), id, name);
        }

        public LicenseTypeBuilder id(String id)
        {
            this.id = checkNotNull(id);
            return this;
        }

        public LicenseTypeBuilder name(String name)
        {
            this.name = checkNotNull(name);
            return this;
        }
    }

    public static class LinksBuilder
    {
        private Map<Pair<String, Option<String>>, URI> links = new HashMap<Pair<String, Option<String>>, URI>();

        public Links build()
        {
            ImmutableList.Builder<Links.Link> items = ImmutableList.builder();
            for (Map.Entry<Pair<String, Option<String>>, URI> entry: links.entrySet())
            {
                items.add(new Links.Link(entry.getKey().first(), entry.getKey().second().getOrElse((String) null),
                                         entry.getValue()));
            }
            return new Links(items.build());
        }

        public LinksBuilder put(String rel, URI uri)
        {
            return put(rel, none(String.class), uri);
        }

        public LinksBuilder put(String rel, Option<String> type, URI uri)
        {
            links.put(pair(checkNotNull(rel), type), checkNotNull(uri));
            return this;
        }

        public LinksBuilder put(Links links)
        {
            for (Links.Link item: links.getItems())
            {
                put(item.getRel(), item.getType(), item.getHref());
            }
            return this;
        }

        public LinksBuilder put(Map<String, URI> links)
        {
            for (Map.Entry<String, URI> item: links.entrySet())
            {
                put(item.getKey(), item.getValue());
            }
            return this;
        }

        public LinksBuilder remove(String rel)
        {
            links.remove(pair(checkNotNull(rel), none(String.class)));
            return this;
        }

        public LinksBuilder remove(String rel, String type)
        {
            links.remove(pair(checkNotNull(rel), some(type)));
            return this;
        }
        
        public LinksBuilder removeAll()
        {
            links.clear();
            return this;
        }
    }

    public static class ModificationBuilder
    {
        private String by = DEFAULT_STRING;
        private Date date = DEFAULT_DATE;
        
        public Modification build()
        {
            return new Modification(by, date);
        }
        
        public ModificationBuilder by(String by)
        {
            this.by = checkNotNull(by);
            return this;
        }
        
        public ModificationBuilder date(Date date)
        {
            this.date = checkNotNull(date);
            return this;
        }
    }
    
    public static class PluginBuilder extends BuilderWithLinks<PluginBuilder>
    {
        private Date creationDate = DEFAULT_DATE;
        private String name = DEFAULT_STRING;
        private String pluginKey = DEFAULT_STRING;
        private Option<String> description = none();
        private Option<String> summary = none();
        private boolean deployable = true;
        private long downloadCount = DEFAULT_LONG;
        private Option<Modification> lastModified = none();
        private Option<Vendor> vendor = none();
        private PluginMedia media = ModelBuilders.pluginMedia().build();
        private ReviewSummary reviewSummary = ModelBuilders.reviewSummary().build();
        private Reviews reviews = ModelBuilders.reviews().build();
        private Collection<PluginCategory> categories = ImmutableList.of();
        private PluginVersion version = ModelBuilders.pluginVersion().build();
        private Collection<PluginVersion> versions = ImmutableList.of();
        private int versionsCount = DEFAULT_INT;
        private boolean isOldVersion = false;
        private Collection<ApplicationSummary> compatibleApplications = ImmutableList.of();
        private Option<Pricing> pricing = none();

        public PluginBuilder()
        {
            links.put("alternate", DEFAULT_URI);
        }
        
        public Plugin build()
        {
            return new Plugin(links.build(), creationDate, name, pluginKey, nullableString(description),
                              nullableString(summary), deployable, downloadCount,
                              lastModified.getOrElse((Modification) null),
                              vendor.getOrElse((Vendor) null),
                              media, reviewSummary, reviews, categories, version,
                              new Plugin.PluginVersions(versions, versionsCount),
                              isOldVersion, compatibleApplications,
                              pricing.getOrElse((Pricing) null));
                              
        }
        
        public PluginBuilder creationDate(Date date)
        {
            this.creationDate = checkNotNull(date);
            return this;
        }
        
        public PluginBuilder name(String name)
        {
            this.name = checkNotNull(name);
            return this;
        }
        
        public PluginBuilder pluginKey(String pluginKey)
        {
            this.pluginKey = checkNotNull(pluginKey);
            return this;
        }
        
        public PluginBuilder description(Option<String> description)
        {
            this.description = checkNotNull(description);
            return this;
        }
        
        public PluginBuilder summary(Option<String> summary)
        {
            this.summary = checkNotNull(summary);
            return this;
        }
        
        public PluginBuilder deployable(boolean deployable)
        {
            this.deployable = deployable;
            return this;
        }
        
        public PluginBuilder downloadCount(long downloadCount)
        {
            this.downloadCount = downloadCount;
            return this;
        }

        public PluginBuilder lastModified(Option<Modification> lastModified)
        {
            this.lastModified = checkNotNull(lastModified);
            return this;
        }

        public PluginBuilder vendor(Option<Vendor> vendor)
        {
            this.vendor = checkNotNull(vendor);
            return this;
        }

        public PluginBuilder media(PluginMedia pluginMedia)
        {
            this.media = checkNotNull(pluginMedia);
            return this;
        }
        
        public PluginBuilder reviewSummary(ReviewSummary reviewSummary)
        {
            this.reviewSummary = checkNotNull(reviewSummary);
            return this;
        }

        public PluginBuilder reviews(Reviews reviews)
        {
            this.reviews = checkNotNull(reviews);
            return this;
        }

        public PluginBuilder categories(Iterable<PluginCategory> categories)
        {
            this.categories = ImmutableList.copyOf(checkNotNull(categories));
            return this;
        }

        public PluginBuilder version(PluginVersion version)
        {
            this.version = checkNotNull(version);
            return this;
        }

        public PluginBuilder versions(Iterable<PluginVersion> versions)
        {
            this.versions = ImmutableList.copyOf(checkNotNull(versions));
            return this;
        }
        
        public PluginBuilder versionsCount(int count)
        {
            this.versionsCount = count;
            return this;
        }
        
        public PluginBuilder oldVersion(boolean isOldVersion)
        {
            this.isOldVersion = isOldVersion;
            return this;
        }
        
        public PluginBuilder compatibleApplications(Iterable<ApplicationSummary> apps)
        {
            this.compatibleApplications = ImmutableList.copyOf(checkNotNull(apps));
            return this;
        }

        public PluginBuilder pricing(Option<Pricing> pricing)
        {
            this.pricing = checkNotNull(pricing);
            return this;
        }
    }
    
    public static class PluginCategoryBuilder extends BuilderWithLinks<PluginCategoryBuilder>
    {
        private String name = DEFAULT_STRING;

        public PluginCategoryBuilder()
        {
            links.put("alternate", DEFAULT_URI);
        }
        
        public PluginCategory build()
        {
            return new PluginCategory(links.build(), name);
        }
        
        public PluginCategoryBuilder name(String name)
        {
            this.name = checkNotNull(name);
            return this;
        }
    }
    
    public static class PluginMediaBuilder
    {
        private Icons icons = ModelBuilders.icons().build();
        private Option<Banner> banner = none();
        private Option<String> youtubeId = none();
        private ImmutableList<Screenshot> screenshots = ImmutableList.of();
    
        public PluginMedia build()
        {
            return new PluginMedia(icons, banner.getOrElse((Banner) null), nullableString(youtubeId), screenshots);
        }
        
        public PluginMediaBuilder icons(Icons icons)
        {
            this.icons = checkNotNull(icons);
            return this;
        }
        
        public PluginMediaBuilder banner(Option<Banner> banner)
        {
            this.banner = checkNotNull(banner);
            return this;
        }
        
        public PluginMediaBuilder youtubeId(Option<String> youtubeId)
        {
            this.youtubeId = checkNotNull(youtubeId);
            return this;
        }
        
        public PluginMediaBuilder screenshots(Iterable<Screenshot> screenshots)
        {
            this.screenshots = ImmutableList.copyOf(screenshots);
            return this;
        }
    }

    public static class PluginSummaryBuilder extends BuilderWithLinks<PluginSummaryBuilder>
    {
        private String name = DEFAULT_STRING;
        private String pluginKey = DEFAULT_STRING;
        private Option<String> summary = none();
        private long downloadCount = DEFAULT_LONG;
        private Option<Modification> lastModified = none();
        private Option<Date> lastReleasedDate = none();
        private Option<Vendor> vendor = none();
        private Icons icons = ModelBuilders.icons().build();
        private ReviewSummary reviewSummary = ModelBuilders.reviewSummary().build();
        private Collection<PluginCategory> categories = ImmutableList.of();
        private PluginVersion version = ModelBuilders.pluginVersion().build();
        private Collection<ApplicationSummary> compatibleApplications = ImmutableList.of();
        private Option<Pricing> pricing = none();

        public PluginSummaryBuilder()
        {
            links.put("alternate", DEFAULT_URI);
        }
        
        public PluginSummary build()
        {
            return new PluginSummary(links.build(), name, pluginKey,
                                     vendor.getOrElse((Vendor) null),
                                     version, icons,
                                     nullableString(summary),
                                     reviewSummary, categories, downloadCount,
                                     lastModified.getOrElse((Modification) null),
                                     lastReleasedDate.getOrElse((Date) null),
                                     compatibleApplications,
                                     pricing.getOrElse((Pricing) null));
                              
        }
        
        public PluginSummaryBuilder name(String name)
        {
            this.name = checkNotNull(name);
            return this;
        }
        
        public PluginSummaryBuilder pluginKey(String pluginKey)
        {
            this.pluginKey = checkNotNull(pluginKey);
            return this;
        }
        
        public PluginSummaryBuilder summary(Option<String> summary)
        {
            this.summary = checkNotNull(summary);
            return this;
        }
        
        public PluginSummaryBuilder downloadCount(long downloadCount)
        {
            this.downloadCount = downloadCount;
            return this;
        }

        public PluginSummaryBuilder lastModified(Option<Modification> lastModified)
        {
            this.lastModified = checkNotNull(lastModified);
            return this;
        }

        public PluginSummaryBuilder lastReleasedDate(Option<Date> lastReleasedDate)
        {
            this.lastReleasedDate = checkNotNull(lastReleasedDate);
            return this;
        }

        public PluginSummaryBuilder vendor(Option<Vendor> vendor)
        {
            this.vendor = checkNotNull(vendor);
            return this;
        }

        public PluginSummaryBuilder icons(Icons icons)
        {
            this.icons = checkNotNull(icons);
            return this;
        }
        
        public PluginSummaryBuilder reviewSummary(ReviewSummary reviewSummary)
        {
            this.reviewSummary = checkNotNull(reviewSummary);
            return this;
        }

        public PluginSummaryBuilder categories(Iterable<PluginCategory> categories)
        {
            this.categories = ImmutableList.copyOf(checkNotNull(categories));
            return this;
        }

        public PluginSummaryBuilder version(PluginVersion version)
        {
            this.version = checkNotNull(version);
            return this;
        }

        public PluginSummaryBuilder compatibleApplications(Iterable<ApplicationSummary> apps)
        {
            this.compatibleApplications = ImmutableList.copyOf(checkNotNull(apps));
            return this;
        }

        public PluginSummaryBuilder pricing(Option<Pricing> pricing)
        {
            this.pricing = checkNotNull(pricing);
            return this;
        }
    }
    
    public static class PluginVersionBuilder extends BuilderWithLinks<PluginVersionBuilder>
    {
        private long buildNumber = DEFAULT_LONG;
        private Date releaseDate = DEFAULT_DATE;
        private Option<String> releasedBy = none();
        private Option<String> releaseFocus = none();
        private boolean stable = true;
        private String status = DEFAULT_STRING;
        private LicenseType license = ModelBuilders.licenseType().build();
        private String version = DEFAULT_STRING;
        private boolean deployable = true;
        private String supportType = DEFAULT_STRING;
        private String pluginSystemVersion = DEFAULT_STRING;
        private Option<String> releaseNotes = none();
        private Option<String> summary = none();
        private Collection<ApplicationSummary> compatibleApplications = ImmutableList.of();
        private Collection<VersionCompatibility> compatibilities = ImmutableList.of();
        private MarketplaceType marketplaceType = MarketplaceType.FREE;
        private String marketplaceTypeTooltip = DEFAULT_STRING;
        private Collection<Screenshot> screenshots = ImmutableList.of();
        private Option<String> youtubeId = none();

        public PluginVersionBuilder()
        {
            links.put("alternate", DEFAULT_URI);
        }
        
        public PluginVersion build()
        {
            return new PluginVersion(links.build(), buildNumber, releaseDate,
                                     nullableString(releasedBy), nullableString(releaseFocus),
                                     stable, status, license, version, deployable, supportType, pluginSystemVersion,
                                     nullableString(releaseNotes), nullableString(summary),
                                     compatibleApplications, compatibilities,
                                     ImmutableMap.of("type", marketplaceType.getDisplayName(), "tooltip", marketplaceTypeTooltip),
                                     screenshots, nullableString(youtubeId));
        }
        
        /**
         * An integer that uniquely identifies this version and determines the sort order of versions.
         */
        public PluginVersionBuilder buildNumber(long buildNumber)
        {
            this.buildNumber = buildNumber;
            return this;
        }
        
        /**
         * True if this version is visible to everyone, false if it is a draft version.  Draft versions
         * can only be seen by an authenticated user who has permission to edit the plugin.
         */
        public PluginVersionBuilder published(boolean published)
        {
            this.status = published ? "Published" : "Draft";
            return this;
        }
        
        /**
         * The date on which the version was made available.
         */
        public PluginVersionBuilder releaseDate(Date date)
        {
            this.releaseDate = checkNotNull(date);
            return this;
        }
        
        /**
         * The user who released this version. This cannot be set with
         * {@link com.atlassian.marketplace.client.api.Plugins#putVersion(String, PluginVersion)};
         * it is based on the authentication credentials used when the version was created.
         */
        public PluginVersionBuilder releasedBy(Option<String> releasedBy)
        {
            this.releasedBy = checkNotNull(releasedBy);
            return this;
        }

        /**
         * A short summary of the nature of the version (major feature changes, bug fixes, etc.).
         */
        public PluginVersionBuilder releaseFocus(Option<String> releaseFocus)
        {
            this.releaseFocus = checkNotNull(releaseFocus);
            return this;
        }
        
        /**
         * True if this is a stable (non-alpha, non-beta) release.
         */
        public PluginVersionBuilder stable(boolean stable)
        {
            this.stable = stable;
            return this;
        }

        /**
         * @deprecated  Use {@link #published(boolean)}.
         */
        public PluginVersionBuilder status(String status)
        {
            this.status = checkNotNull(status);
            return this;
        }
        
        /**
         * Information about how this version is licensed.
         */
        public PluginVersionBuilder license(LicenseType license)
        {
            this.license = checkNotNull(license);
            return this;
        }

        /**
         * The version number string (e.g. "1.0.0");
         */
        public PluginVersionBuilder version(String version)
        {
            this.version = checkNotNull(version);
            return this;
        }

        /**
         * True if this version can be installed directly via the Plugin Manager.
         */
        public PluginVersionBuilder deployable(boolean deployable)
        {
            this.deployable = deployable;
            return this;
        }

        /**
         * A short string describing what type of support is available (e.g. "Vendor").
         */
        public PluginVersionBuilder supportType(String supportType)
        {
            this.supportType = checkNotNull(supportType);
            return this;
        }

        /**
         * The Atlassian plugins framework version of the plugin (e.g. "Two").
         */
        public PluginVersionBuilder pluginSystemVersion(String pluginSystemVersion)
        {
            this.pluginSystemVersion = checkNotNull(pluginSystemVersion);
            return this;
        }
        
        /**
         * Optional release notes for this version.
         */
        public PluginVersionBuilder releaseNotes(Option<String> releaseNotes)
        {
            this.releaseNotes = checkNotNull(releaseNotes);
            return this;
        }
        
        /**
         * An optional short description of this version.
         */
        public PluginVersionBuilder summary(Option<String> summary)
        {
            this.summary = checkNotNull(summary);
            return this;
        }

        /**
         * The applications compatible with this plugin version. This cannot be set with
         * {@link com.atlassian.marketplace.client.api.Plugins#putVersion(String, PluginVersion)};
         * instead use {@link #compatibilities}.
         */
        public PluginVersionBuilder compatibleApplications(Iterable<ApplicationSummary> apps)
        {
            this.compatibleApplications = ImmutableList.copyOf(checkNotNull(apps));
            return this;
        }

        /**
         * The specific application version(s) compatible with this plugin version. If you are
         * adding or updating a version with {@link com.atlassian.marketplace.client.api.Plugins#putVersion(String, PluginVersion)},
         * you only need to set the minimum and maximum build numbers; the version strings are ignored.
         */
        public PluginVersionBuilder compatibilities(Iterable<VersionCompatibility> compatibilities)
        {
            this.compatibilities = ImmutableList.copyOf(checkNotNull(compatibilities));
            return this;
        }

        /**
         * Indicates which payment system, if any, is used for buying licenses for the plugin.
         */
        public PluginVersionBuilder marketplaceType(MarketplaceType marketplaceType)
        {
            this.marketplaceType = checkNotNull(marketplaceType);
            return this;
        }

        /**
         * A human-readable explanation of the value of {@link #getMarketplaceType()}. This cannot be set with
         * {@link com.atlassian.marketplace.client.api.Plugins#putVersion(String, PluginVersion)}.
         */
        public PluginVersionBuilder marketplaceTypeTooltip(String marketplaceTypeTooltip)
        {
            this.marketplaceTypeTooltip = checkNotNull(marketplaceTypeTooltip);
            return this;
        }
        
        /**
         * The screenshot images associated with this version, if any. This cannot be set with
         * {@link com.atlassian.marketplace.client.api.Plugins#putVersion(String, PluginVersion)}.
         */
        public PluginVersionBuilder screenshots(Iterable<Screenshot> screenshots)
        {
            this.screenshots = ImmutableList.copyOf(checkNotNull(screenshots));
            return this;
        }
        
        /**
         * The YouTube video key associated with this version, if any.
         */
        public PluginVersionBuilder youtubeId(Option<String> youtubeId)
        {
            this.youtubeId = checkNotNull(youtubeId);
            return this;
        }
        
        /**
         * Shortcut for getting the "binary" link in the links map.  This is the URI from which the
         * plugin version can be downloaded.
         */
        public PluginVersionBuilder binaryUri(Option<URI> uri)
        {
            if (!uri.isDefined())
            {
                links.remove("binary");
            }
            for (URI u: uri)
            {
                links.put("binary", u);
            }
            return this;
        }
    }

    public static class PricingBuilder extends BuilderWithLinks<PricingBuilder>
    {
        private Collection<PricingItem> items = ImmutableList.of();
        private boolean expertDiscountOptOut = false;
        private Option<String> parent = none();
        private Option<Date> lastModified = none();
        private Collection<PricingItem> legacyItems = ImmutableList.of();
        private Option<String> nonLegacyStartVersion = none();
        private Option<String> legacyMaxVersion = none();
        
        public Pricing build()
        {
            return new Pricing(links.build(), items, expertDiscountOptOut, nullableString(parent),
                               lastModified.getOrElse((Date) null), legacyItems,
                               nullableString(nonLegacyStartVersion), nullableString(legacyMaxVersion));
        }

        public PricingBuilder items(Iterable<PricingItem> items)
        {
            this.items = ImmutableList.copyOf(checkNotNull(items));
            return this;
        }
        
        public PricingBuilder expertDiscountOptOut(boolean optOut)
        {
            this.expertDiscountOptOut = optOut;
            return this;
        }
        
        public PricingBuilder parent(Option<String> parent)
        {
            this.parent = checkNotNull(parent);
            return this;
        }
        
        public PricingBuilder lastModified(Option<Date> lastModified)
        {
            this.lastModified = checkNotNull(lastModified);
            return this;
        }
        
        public PricingBuilder legacyItems(Iterable<PricingItem> legacyItems)
        {
            this.legacyItems = ImmutableList.copyOf(checkNotNull(legacyItems));
            return this;
        }
        
        public PricingBuilder nonLegacyStartVersion(Option<String> nonLegacyStartVersion)
        {
            this.nonLegacyStartVersion = checkNotNull(nonLegacyStartVersion);
            return this;
        }
        
        public PricingBuilder legacyMaxVersion(Option<String> legacyMaxVersion)
        {
            this.legacyMaxVersion = checkNotNull(legacyMaxVersion);
            return this;
        }
    }
    
    public static class PricingItemBuilder
    {
        private long id = DEFAULT_LONG;
        private Option<String> description = none();
        private Option<String> unitDescription = none();
        private float usdAmount = DEFAULT_PRICE;
        private int unitCount = DEFAULT_INT;
        private boolean enterprise = false;
        private boolean starter = false;
        private String saleType = DEFAULT_STRING;
        private String licenseType = DEFAULT_STRING;
        private boolean legacyDisplay = false;

        public PricingItem build()
        {
            return new PricingItem(id, nullableString(description), nullableString(unitDescription),
                                   usdAmount, unitCount, enterprise, starter, saleType, licenseType, legacyDisplay);
        }
        
        public PricingItemBuilder id(long id)
        {
            this.id = id;
            return this;
        }
        
        public PricingItemBuilder description(Option<String> description)
        {
            this.description = checkNotNull(description);
            return this;
        }
        
        public PricingItemBuilder unitDescription(Option<String> unitDescription)
        {
            this.unitDescription = checkNotNull(unitDescription);
            return this;
        }
        
        public PricingItemBuilder usdAmount(float usdAmount)
        {
            this.usdAmount = usdAmount;
            return this;
        }

        public PricingItemBuilder unitCount(int unitCount)
        {
            this.unitCount = unitCount;
            return this;
        }

        public PricingItemBuilder enterprise(boolean enterprise)
        {
            this.enterprise = enterprise;
            return this;
        }

        public PricingItemBuilder starter(boolean starter)
        {
            this.starter = starter;
            return this;
        }

        public PricingItemBuilder saleType(String saleType)
        {
            this.saleType = checkNotNull(saleType);
            return this;
        }

        public PricingItemBuilder licenseType(String licenseType)
        {
            this.licenseType = checkNotNull(licenseType);
            return this;
        }

        public PricingItemBuilder legacyDisplay(boolean legacyDisplay)
        {
            this.legacyDisplay = legacyDisplay;
            return this;
        }
    }

    public static class ReviewBuilder extends BuilderWithLinks<ReviewBuilder>
    {
        private String pluginName = DEFAULT_STRING;
        private Option<UserSummary> author = none();
        private int stars = DEFAULT_STARS;
        private Date date = DEFAULT_DATE;
        private Option<String> review = none();
        private Option<ReviewResponse> response = none();
        
        public Review build()
        {
            return new Review(links.build(), pluginName, author.getOrElse((UserSummary) null), stars,
                              date, nullableString(review), response.getOrElse((ReviewResponse) null));
        }
        
        public ReviewBuilder pluginName(String pluginName)
        {
            this.pluginName = checkNotNull(pluginName);
            return this;
        }
        
        public ReviewBuilder author(Option<UserSummary> author)
        {
            this.author = checkNotNull(author);
            return this;
        }
        
        public ReviewBuilder stars(int stars)
        {
            this.stars = stars;
            return this;
        }
        
        public ReviewBuilder date(Date date)
        {
            this.date = checkNotNull(date);
            return this;
        }
        
        public ReviewBuilder review(Option<String> review)
        {
            this.review = checkNotNull(review);
            return this;
        }
        
        public ReviewBuilder response(Option<ReviewResponse> response)
        {
            this.response = checkNotNull(response);
            return this;
        }
    }
    
    public static class ReviewResponseBuilder extends BuilderWithLinks<ReviewResponseBuilder>
    {
        private String response = DEFAULT_STRING;
        
        public ReviewResponse build()
        {
            return new ReviewResponse(links.build(), response);
        }
        
        public ReviewResponseBuilder response(String response)
        {
            this.response = checkNotNull(response);
            return this;
        }
    }
    
    public static class ReviewsBuilder extends BuilderWithLinks<ReviewsBuilder>
    {
        private Collection<Review> reviews = ImmutableList.of();
        
        public Reviews build()
        {
            return new Reviews(links.build(), reviews);
        }

        public ReviewsBuilder reviews(Iterable<Review> reviews)
        {
            this.reviews = ImmutableList.copyOf(checkNotNull(reviews));
            return this;
        }
    }

    public static class ReviewSummaryBuilder extends BuilderWithLinks<ReviewSummaryBuilder>
    {
        private float averageStars = DEFAULT_STARS;
        private int count = DEFAULT_INT;
        
        public ReviewSummary build()
        {
            return new ReviewSummary(links.build(), averageStars, count);
        }
        
        public ReviewSummaryBuilder averageStars(float averageStars)
        {
            this.averageStars = averageStars;
            return this;
        }
        
        public ReviewSummaryBuilder count(int count)
        {
            this.count = count;
            return this;
        }
    }
    
    public static class ScreenshotBuilder
    {
        private String name = DEFAULT_STRING;
        private Image image = ModelBuilders.image().build();
        private Option<Image> carouselImage = none();
        
        public Screenshot build()
        {
            return new Screenshot(name, image, carouselImage.getOrElse((Image) null));
        }
        
        public ScreenshotBuilder name(String name)
        {
            this.name = checkNotNull(name);
            return this;
        }

        public ScreenshotBuilder image(Image image)
        {
            this.image = checkNotNull(image);
            return this;
        }

        public ScreenshotBuilder carouselImage(Option<Image> carouselImage)
        {
            this.carouselImage = checkNotNull(carouselImage);
            return this;
        }
    }
    
    public static class UserSummaryBuilder extends BuilderWithLinks<UserSummaryBuilder>
    {
        private long id = DEFAULT_LONG;
        private String name = DEFAULT_STRING;
        private Option<String> email = none();
        
        public UserSummary build()
        {
            return new UserSummary(links.build(), id, name, nullableString(email));
        }
        
        public UserSummaryBuilder id(long id)
        {
            this.id = id;
            return this;
        }
        
        public UserSummaryBuilder name(String name)
        {
            this.name = checkNotNull(name);
            return this;
        }
        
        public UserSummaryBuilder email(Option<String> email)
        {
            this.email = checkNotNull(email);
            return this;
        }
    }
    
    public static class VendorBuilder extends BuilderWithLinks<VendorBuilder>
    {
        private long id = DEFAULT_LONG;
        private String name = DEFAULT_STRING;
        private Option<String> email = none();
        private Option<String> phone = none();
        private Option<String> otherContactDetails = none();
        
        public Vendor build()
        {
            return new Vendor(links.build(), id, name, nullableString(email), nullableString(phone),
                              nullableString(otherContactDetails));
        }
        
        public VendorBuilder id(long id)
        {
            this.id = id;
            return this;
        }
        
        public VendorBuilder name(String name)
        {
            this.name = checkNotNull(name);
            return this;
        }
        
        public VendorBuilder email(Option<String> email)
        {
            this.email = checkNotNull(email);
            return this;
        }
        
        public VendorBuilder phone(Option<String> phone)
        {
            this.phone = checkNotNull(phone);
            return this;
        }
        
        public VendorBuilder otherContactDetails(Option<String> otherContactDetails)
        {
            this.otherContactDetails = checkNotNull(otherContactDetails);
            return this;
        }
    }

    public static class VersionCompatibilityBuilder
    {
        private String applicationName = DEFAULT_STRING;
        private VersionCompatibility.VersionAndBuildNumber min =
            new VersionCompatibility.VersionAndBuildNumber(DEFAULT_STRING, 0L);
        private VersionCompatibility.VersionAndBuildNumber max =
            new VersionCompatibility.VersionAndBuildNumber(DEFAULT_STRING, Long.MAX_VALUE);
        
        public VersionCompatibility build()
        {
            return new VersionCompatibility(applicationName, min, max);
        }
        
        public VersionCompatibilityBuilder applicationName(String applicationName)
        {
            this.applicationName = checkNotNull(applicationName);
            return this;
        }
        
        public VersionCompatibilityBuilder min(VersionCompatibility.VersionAndBuildNumber min)
        {
            this.min = checkNotNull(min);
            return this;
        }
        
        public VersionCompatibilityBuilder max(VersionCompatibility.VersionAndBuildNumber max)
        {
            this.max = checkNotNull(max);
            return this;
        }
    }
    
    private static String nullableString(Option<String> s)
    {
        return s.getOrElse((String) null);
    }
}
