package com.atlassian.marketplace.client.model;

import java.net.URI;
import java.util.Collection;

import com.atlassian.upm.api.util.Option;

import com.google.common.collect.ImmutableList;

import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.annotate.JsonProperty;
import org.codehaus.jackson.annotate.JsonValue;

import static com.atlassian.marketplace.client.util.ModelUtil.requireProperty;
import static com.atlassian.upm.api.util.Option.none;
import static com.atlassian.upm.api.util.Option.option;
import static com.atlassian.upm.api.util.Option.some;

public final class Links
{
    public static final String REST_TYPE = "application/json";
    public static final String WEB_TYPE = "text/html";

	private final Collection<Link> items;

	@JsonCreator
	Links(Collection<Link> items)
	{
		this.items = ImmutableList.copyOf(items);
	}

	@JsonValue
	public Collection<Link> getItems()
	{
		// This has to use Collection rather than Iterable because of how @JsonValue works.
		return items;		
	}

	public Option<URI> get(String rel)
	{
		for (Link link: items)
		{
			if (rel.equalsIgnoreCase(link.getRel()))
			{
				return some(link.getHref());
			}
		}
		return none();
	}

	public Option<URI> get(String rel, String contentType)
	{
		for (Link link: items)
		{
			if (rel.equalsIgnoreCase(link.getRel()) && contentType.equalsIgnoreCase(link.getType().getOrElse("")))
			{
				return some(link.getHref());
			}
		}
		return none();
	}

	public Option<String> getContentType(String rel)
	{
        for (Link link: items)
        {
            if (rel.equalsIgnoreCase(link.getRel()))
            {
                return link.getType();
            }
        }
        return none();
	}
	
	public static final class Link
	{
		@JsonProperty private final String rel;
		@JsonProperty private final String type;
		@JsonProperty private final URI href;

		@JsonCreator
		Link(@JsonProperty("rel") String rel, @JsonProperty("type") String type,
		     @JsonProperty("href") URI href)
        {
            this.rel = requireProperty(rel, "rel");
            this.type = type;  // optional
            this.href = requireProperty(href, "href");
        }

        public String getRel()
        {
            return rel;
        }

        @JsonIgnore
        public Option<String> getType()
        {
            return option(type);
        }

        public URI getHref()
        {
            return href;
        }
	}
}
