package com.atlassian.marketplace.client.model;

import java.util.Collection;

import com.atlassian.marketplace.client.api.ApplicationKey;

import com.google.common.collect.ImmutableList;

import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.annotate.JsonProperty;

import static com.atlassian.marketplace.client.util.ModelUtil.requireList;
import static com.atlassian.marketplace.client.util.ModelUtil.requireProperty;

/**
 * Information about an Atlassian application.
 */
public final class Application
{
    @JsonProperty private final Links links;
    @JsonProperty private final String key;
    @JsonProperty private final String name;
    @JsonProperty private final int order;
    @JsonProperty private final int pluginCount;
    @JsonProperty private final Collection<ApplicationVersion> versions;
    @JsonProperty private final Collection<String> categories;
    
    // fields omitted: status
    
    @JsonCreator
    Application(@JsonProperty("links") Links links,
                @JsonProperty("key") String key,
                @JsonProperty("name") String name,
                @JsonProperty("order") Integer order,
                @JsonProperty("pluginCount") Integer pluginCount,
                @JsonProperty("versions") Collection<ApplicationVersion> versions,
                @JsonProperty("categories") Collection<String> categories)
    {
        this.links = requireProperty(links, "links");
        this.key = requireProperty(key, "key");
        this.name = requireProperty(name, "name");
        this.order = requireProperty(order, "order");
        this.pluginCount = requireProperty(pluginCount, "pluginCount");
        this.versions = requireList(versions, "versions");
        this.categories = requireList(categories, "categories");
    }

    public Links getLinks()
    {
        return links;
    }

    /**
     * The short unique identifier for the application, corresponding to one of the constants in
     * {@link ApplicationKey}.
     */
    @JsonIgnore
    public ApplicationKey getKey()
    {
        return ApplicationKey.valueOf(key);
    }

    /**
     * The application name.
     */
    public String getName()
    {
        return name;
    }

    /**
     * The order in which this application is listed on the Marketplace.
     */
    public int getOrder()
    {
        return order;
    }

    /**
     * The total number of published plugins that are compatible with this application.
     */
    public int getPluginCount()
    {
        return pluginCount;
    }

    /**
     * All published versions of this application.
     */
    @JsonIgnore
    public Iterable<ApplicationVersion> getVersions()
    {
        return ImmutableList.copyOf(versions);
    }

    /**
     * All of the plugin categories in which there are any published plugins compatible with this
     * application.
     */
    @JsonIgnore
    public Iterable<String> getCategories()
    {
        return ImmutableList.copyOf(categories);
    }
}
