package com.atlassian.marketplace.client.impl;

import java.util.Collection;
import java.util.Date;

import com.atlassian.marketplace.client.MpacException;
import com.atlassian.marketplace.client.api.ApplicationKey;
import com.atlassian.marketplace.client.api.BannerQuery;
import com.atlassian.marketplace.client.api.Page;
import com.atlassian.marketplace.client.api.PageReference;
import com.atlassian.marketplace.client.api.PluginDetailQuery;
import com.atlassian.marketplace.client.api.PluginDetailQuery.Version;
import com.atlassian.marketplace.client.api.PluginQuery;
import com.atlassian.marketplace.client.api.Plugins;
import com.atlassian.marketplace.client.api.PricingQuery;
import com.atlassian.marketplace.client.impl.representations.PluginBannersRepresentation;
import com.atlassian.marketplace.client.impl.representations.PluginsRepresentation;
import com.atlassian.marketplace.client.impl.representations.RootRepresentation;
import com.atlassian.marketplace.client.model.Banner;
import com.atlassian.marketplace.client.model.LicenseType;
import com.atlassian.marketplace.client.model.Links;
import com.atlassian.marketplace.client.model.Plugin;
import com.atlassian.marketplace.client.model.PluginSummary;
import com.atlassian.marketplace.client.model.PluginVersion;
import com.atlassian.marketplace.client.model.VersionCompatibility;
import com.atlassian.marketplace.client.util.UriBuilder;
import com.atlassian.upm.api.util.Option;

import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;

import org.codehaus.jackson.annotate.JsonProperty;

import static com.atlassian.marketplace.client.impl.representations.RepresentationLinks.PLUGINS_REL;
import static com.google.common.collect.Iterables.transform;

final class PluginsImpl implements Plugins
{
    /**
     * Temporary, because the server doesn't currently tell us what limit it actually used if
     * we omit that parameter, and the client needs a value for this in order to implement paging.
     */
    static final int DEFAULT_LIMIT = 10;
    
    private final DefaultMarketplaceClient client;
    private final RootRepresentation root;

    PluginsImpl(DefaultMarketplaceClient client, RootRepresentation root)
    {
        this.client = client;
        this.root = root;
    }

    @Override
    public Page<PluginSummary> find(PluginQuery query) throws MpacException
    {
        final UriBuilder uri = getQueryBaseUri(query);
        addAppParams(uri, query.getApplication(), query.getAppBuildNumber());
        for (String c: query.getCategories())
        {
            uri.queryParam("category", c);
        }
        for (PluginQuery.Cost cost: query.getCost())
        {
            uri.queryParam("cost", cost.getKey());
        }
        for (PricingQuery pricing: query.getIncludePricing())
        {
            uri.queryParam("pricing", "true");
            for (String licenseType: pricing.getLicenseType())
            {
                uri.queryParam("licenseType", licenseType);
            }
        }
        addOffsetLimitParams(uri, query.getOffset(), query.getLimit());
        return getMorePlugins(new PageReference<PluginSummary>(uri.build(), query.getOffset(), query.getLimit()));
    }

    @Override
    public Page<PluginSummary> getMorePlugins(PageReference<PluginSummary> page) throws MpacException
    {
        PluginsRepresentation pluginsRep = client.getEntity(client.resolveLink(page.getUri()), PluginsRepresentation.class);
        return new PageImpl<PluginSummary>(page, pluginsRep.getLinks(), pluginsRep.getPlugins(), pluginsRep.getCount());
    }
    
    @Override
    public Option<Plugin> get(PluginDetailQuery query) throws MpacException
    {
        final UriBuilder uri = getDetailQueryBaseUri(query);
        for (ApplicationKey appKey: query.getApplication())
        {
            uri.queryParam("application", appKey.getKey());
            for (Long appBuild: query.getAppBuildNumber())
            {
                uri.queryParam("buildNumber", appBuild);
            }
        }
        for (PricingQuery pricing: query.getIncludePricing())
        {
            uri.queryParam("pricing", "true");
            for (String licenseType: pricing.getLicenseType())
            {
                uri.queryParam("licenseType", licenseType);
            }
        }
        for (Integer limitVersions: query.getLimitVersions())
        {
            uri.queryParam("limit-versions", limitVersions);
        }
        for (Integer limitReviews: query.getLimitReviews())
        {
            uri.queryParam("limit-reviews", limitReviews);
        }
        return client.getOptionalEntity(uri.build(), Plugin.class);
    }
    
    @Override
    public Page<Banner> findBanners(BannerQuery query) throws MpacException
    {
        final UriBuilder uri = getPluginsLink("banners");
        addAppParams(uri, query.getApplication(), query.getAppBuildNumber());
        for (String label: query.getLabel())
        {
            uri.queryParam("marketingLabel", label);
        }
        addOffsetLimitParams(uri, query.getOffset(), query.getLimit());
        return getMoreBanners(new PageReference<Banner>(uri.build(), query.getOffset(), query.getLimit()));
    }
    
    @Override
    public Page<Banner> getMoreBanners(PageReference<Banner> page) throws MpacException
    {
        PluginBannersRepresentation bannersRep = client.getEntity(client.resolveLink(page.getUri()), PluginBannersRepresentation.class);
        return new PageImpl<Banner>(page, bannersRep.getLinks(), bannersRep.getBanners(), bannersRep.getCount());
    }
    
    @Override
    public void putVersion(String pluginKey, PluginVersion version) throws MpacException
    {
        client.putEntity(getVersionUri(pluginKey, version.getBuildNumber()).build(), new PluginVersionPutData(version));
    }
    
    private UriBuilder getQueryBaseUri(PluginQuery query) throws MpacException
    {
        for (String searchText: query.getSearchText())
        {
            // Use the advanced search resource (which doesn't allow filtering by kind).
            // There's currently no REST link for this, so use a hard-coded URL path (sorry).
            return pluginsBaseUri().path("search").queryParam("q", searchText); 
        }
        for (PluginQuery.View view: query.getView())
        {
            // Follow the link from the base plugins resource to find the by-kind plugins view
            return getPluginsLink(view.getKey());
        }
        return pluginsBaseUri();
    }

    private UriBuilder getDetailQueryBaseUri(PluginDetailQuery query) throws MpacException
    {
        for (Version version: query.getVersion())
        {
            UriBuilder ret = pluginsBaseUri().path(client.urlEncode(query.getPluginKey()))
                .path("version")
                .path(client.urlEncode(version.getVersion()));
            return version.isGreaterThan() ? ret.path("update") : ret; 
        }
        return pluginsBaseUri().path(query.getPluginKey());
    }

    private UriBuilder getVersionUri(String pluginKey, long buildNumber) throws MpacException
    {
        return pluginsBaseUri().path(client.urlEncode(pluginKey)).path("build-number").path(String.valueOf(buildNumber));
    }
    
    private void addAppParams(UriBuilder uri, Option<ApplicationKey> appKey, Option<Long> appBuild)
    {
        for (ApplicationKey a: appKey)
        {
            uri.queryParam("application", a.getKey());
            for (Long b: appBuild)
            {
                uri.queryParam("buildNumber", b);
            }
        }
    }
    
    private void addOffsetLimitParams(UriBuilder uri, int offset, Option<Integer> limit)
    {
        if (offset > 0)
        {
            uri.queryParam("offset", offset);
        }
        for (Integer l: limit)
        {
            uri.queryParam("limit", l);
        }
    }

    private UriBuilder pluginsBaseUri() throws MpacException
    {
        return UriBuilder.fromUri(client.requireLinkUri(root.getLinks(), PLUGINS_REL, root.getClass()));
    }
    
    /**
     * Query the /plugins resource with a limit of zero.  Use this to get the links map without
     * actually getting any plugins.
     */
    private PluginsRepresentation pluginsBaseResource() throws MpacException
    {
        return client.getEntity(pluginsBaseUri().queryParam("limit", 0).build(), PluginsRepresentation.class);
    }
    
    private UriBuilder getPluginsLink(String rel) throws MpacException
    {
        PluginsRepresentation pluginsBase = pluginsBaseResource();
        return UriBuilder.fromUri(client.requireLinkUri(pluginsBase.getLinks(), rel, PluginsRepresentation.class));
    }
    
    static final class PluginVersionPutData
    {
        @JsonProperty final Collection<CompatibilityPutData> compatibilities;
        @JsonProperty final boolean deployable;
        @JsonProperty final String focus;
        @JsonProperty final LicenseType license;
        @JsonProperty final Links links;
        @JsonProperty final String marketplaceType;
        @JsonProperty final String pluginSystemVersion;
        @JsonProperty final Date releaseDate;
        @JsonProperty final String releaseNotes;
        @JsonProperty final boolean stable;
        @JsonProperty final String status;
        @JsonProperty final String summary;
        @JsonProperty final String supportType;
        @JsonProperty final String version;
        @JsonProperty final String youtubeId;

        PluginVersionPutData(PluginVersion from)
        {
            this.compatibilities = ImmutableList.copyOf(transform(from.getCompatibilities(), toCompatibilityPutData()));
            this.deployable = from.isDeployable();
            this.focus = from.getReleaseFocus().getOrElse((String) null);
            this.license = from.getLicense();
            this.links = from.getLinks();
            this.marketplaceType = from.getMarketplaceType().getDisplayName();
            this.pluginSystemVersion = from.getPluginSystemVersion();
            this.supportType = from.getSupportType();
            this.releaseDate = from.getReleaseDate();
            this.releaseNotes = from.getReleaseNotes().getOrElse((String) null);
            this.stable = from.isStable();
            this.status = from.getStatus();
            this.summary = from.getSummary().getOrElse((String) null);
            this.version = from.getVersion();
            this.youtubeId = from.getYoutubeId().getOrElse((String) null);
        }
        
        private Function<VersionCompatibility, CompatibilityPutData> toCompatibilityPutData()
        {
            return new Function<VersionCompatibility, PluginsImpl.PluginVersionPutData.CompatibilityPutData>()
            {
                public CompatibilityPutData apply(VersionCompatibility from)
                {
                    return new CompatibilityPutData(from);
                }
            };
        }
        static final class CompatibilityPutData
        {
            @JsonProperty final String applicationName;
            @JsonProperty final long min;
            @JsonProperty final long max;
            
            CompatibilityPutData(VersionCompatibility from)
            {
                this.applicationName = from.getApplicationName();
                this.min = from.getMin().getBuildNumber();
                this.max = from.getMax().getBuildNumber();
            }
        }
    }
}
