package com.atlassian.marketplace.client.api;

import com.atlassian.marketplace.client.HttpConfiguration;
import com.atlassian.marketplace.client.MpacException;
import com.atlassian.marketplace.client.model.Banner;
import com.atlassian.marketplace.client.model.Plugin;
import com.atlassian.marketplace.client.model.PluginSummary;
import com.atlassian.marketplace.client.model.PluginVersion;
import com.atlassian.upm.api.util.Option;

/**
 * Starting point for all resources that return plugin information.
 */
public interface Plugins
{
    /**
     * Queries a list of plugins.  The search filter and ordering are determined by the {@link PluginQuery}.
     * Search results are always limited to one page at a time; the number of items per page is determined
     * by the server.
     * @param query  a {@link PluginQuery}
     * @return  a {@link Page} containing the search results
     * @throws MpacException  if the server was unavailable or returned an error
     * @see #getMorePlugins
     */
    Page<PluginSummary> find(PluginQuery query) throws MpacException;
    
    /**
     * Queries a page of a result set from a previous plugin query.  The {@link PageReference} is obtained
     * by calling {@link Page#getPrevious()}, {@link Page#getNext()}, or {@link Page#getReference()} on
     * an existing {@link Page} of plugin summaries.
     * @param query  a {@link PageReference} obtained from a previous result page
     * @return  a {@link Page} containing the search results
     * @throws MpacException  if the server was unavailable or returned an error
     * @see #find
     */
    Page<PluginSummary> getMorePlugins(PageReference<PluginSummary> page) throws MpacException;
    
    /**
     * Queries the details for a single plugin.
     * <p>
     * The returned plugin's {@link Plugin#getVersion} and {@link Plugin#getVersions} properties behave
     * differently depending on the query properties, as follows:
     * <ul>
     * <li> If you specified a version with {@link PluginDetailQuery.Builder#version(Option)}, then
     * {@link Plugin#getVersions} returns an empty collection and {@link Plugin#getVersion} returns the
     * specified version.
     * <li> If you specified {@link PluginDetailQuery.Builder#application(Option)} and, optionally,
     * {@link PluginDetailQuery.Builder#appBuildNumber(Option)}, then {@link Plugin#getVersions} returns
     * a collection of only the versions compatible with that application (or that combination of
     * application and build number), and {@link Plugin#getVersion} returns the latest such version.
     * <li> If you did not specify any of the above, then {@link Plugin
     * If you do not specify any {@link PluginDetailQuery} properties other than the plugin key, then
     * {@link Plugin#getVersions} returns every published version of the plugin, and {@link Plugin#getVersion}
     * returns the latest published version.
     * </ul>
     * @param query  a {@link PluginDetailQuery}
     * @return  a {@link Plugin}, or {@link Option#none()} if the plugin key was not found (or, in the
     *   case of a version-limited query, no matching version was found)
     * @throws MpacException  if the server was unavailable or returned an error
     */
    Option<Plugin> get(PluginDetailQuery query) throws MpacException;
    
    /**
     * Queries plugin banner images.
     * @param query  a {@link BannerQuery}
     * @return  a {@link Page} containing the search results
     * @throws MpacException  if the server was unavailable or returned an error
     * @see #getMoreBanners
     */
    Page<Banner> findBanners(BannerQuery query) throws MpacException;

    /**
     * Queries a page of a result set from a previous banner query.  The {@link PageReference} is obtained
     * by calling {@link Page#getPrevious()}, {@link Page#getNext()}, or {@link Page#getReference()} on
     * an existing {@link Page} of banners.
     * @param query  a {@link PageReference} obtained from a previous result page
     * @return  a {@link Page} containing the search results
     * @throws MpacException  if the server was unavailable or returned an error
     * @see #findBanner
     */
    Page<Banner> getMoreBanners(PageReference<Banner> page) throws MpacException;
    
    /**
     * Adds or updates a plugin version in the Marketplace database.  You can only do this if you have
     * provided credentials in your {@link HttpConfiguration} for a user account that has permission to edit
     * the plugin.
     * <p>
     * The unique identifier of a version is the {@link PluginVersion#getBuildNumber() build number}: if you
     * provide a {@link PluginVersion} with the same build number as an existing version, the existing
     * information will be overwritten, otherwise a new version will be added.  The build number of an
     * existing version cannot be changed.
     * <p>
     * To construct the {@link PluginVersion} object, use {@link com.atlassian.marketplace.client.model.ModelBuilders#pluginVersion()}.
     * Note that some properties of {@link PluginVersion} cannot be updated through this API; these are
     * indicated by comments on the setter methods of {@link com.atlassian.marketplace.client.model.ModelBuilders.PluginVersionBuilder}.
     * 
     * @param pluginKey  the plugin key (the plugin must already exist)
     * @param version  a {@link PluginVersion} containing the new or updated version information
     * @throws MpacException  if the server was unavailable or returned an error
     */
    void putVersion(String pluginKey, PluginVersion version) throws MpacException;
}
