package com.atlassian.marketplace.client.api;

import com.atlassian.marketplace.client.model.PluginSummary;
import com.atlassian.upm.api.util.Option;

import static com.atlassian.upm.api.util.Option.none;
import static com.atlassian.upm.api.util.Option.some;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Encapsulates plugin search parameters that can be passed to {@link Plugins#get(PluginDetailQuery)}.
 */
public final class PluginDetailQuery
{
    public static final class Version
    {
        private final String version;
        private final boolean greaterThan;
        
        private Version(String version, boolean greaterThan)
        {
            this.version = checkNotNull(version);
            this.greaterThan = greaterThan;
        }
        
        public static Version equalTo(String version)
        {
            return new Version(version, false);
        }
        
        public static Version greaterThan(String version)
        {
            return new Version(version, true);
        }
        
        public String getVersion()
        {
            return version;
        }
        
        public boolean isGreaterThan()
        {
            return greaterThan;
        }
        
        @Override
        public String toString()
        {
            return (greaterThan ? "> " : "== ") + version;
        }
        
        @Override
        public boolean equals(Object other)
        {
            return (other instanceof Version) ? toString().equals(other.toString()) : false;
        }
        
        @Override
        public int hashCode()
        {
            return toString().hashCode();
        }
    }
    
    private final String pluginKey;
    private final Option<Version> version;
    private final Option<ApplicationKey> application;
    private final Option<Long> appBuildNumber;
    private final Option<PricingQuery> includePricing;
    private final Option<Integer> limitVersions;
    private final Option<Integer> limitReviews;
    
    /**
     * Returns a new {@link Builder} for constructing a PluginDetailQuery.
     * @param pluginKey  the plugin key (required)
     */
    public static Builder builder(String pluginKey)
    {
        return new Builder(pluginKey);
    }
    
    /**
     * Shortcut for constructing a detail query for a plugin whose summary you already have.
     */
    public static Builder builder(PluginSummary pluginSummary)
    {
        return builder(pluginSummary.getPluginKey());
    }
    
    private PluginDetailQuery(Builder builder)
    {
        pluginKey = builder.pluginKey;
        version = builder.version;
        application = builder.application;
        appBuildNumber = builder.appBuildNumber;
        includePricing = builder.includePricing;
        limitVersions = builder.limitVersions;
        limitReviews = builder.limitReviews;
    }
    
    public String getPluginKey()
    {
        return pluginKey;
    }
    
    public Option<Version> getVersion()
    {
        return version;
    }
    
    public Option<ApplicationKey> getApplication()
    {
        return application;
    }

    public Option<Long> getAppBuildNumber()
    {
        return appBuildNumber;
    }

    public Option<PricingQuery> getIncludePricing()
    {
        return includePricing;
    }
    
    public Option<Integer> getLimitVersions()
    {
        return limitVersions;
    }
    
    public Option<Integer> getLimitReviews()
    {
        return limitReviews;
    }
    
    @Override
    public String toString()
    {
        StringBuilder b = new StringBuilder();
        b.append("PluginDetailQuery(").append(pluginKey);
        for (Version v: version)
        {
            b.append(", version(").append(v.toString()).append(")");
        }
        for (ApplicationKey a: application)
        {
            b.append(", application(").append(a.getKey()).append(")"); 
        }
        for (Long ab: appBuildNumber)
        {
            b.append(", appBuildNumber(").append(ab).append(")");
        }
        for (PricingQuery pq: includePricing)
        {
            b.append(", pricing(").append(pq).append(")");
        }
        for (Integer lv: limitVersions)
        {
            b.append(", limitVersions(").append(lv).append(")");
        }
        for (Integer lr: limitReviews)
        {
            b.append(", limitReviews(").append(lr).append(")");
        }
        return b.append(")").toString();
    }
    
    @Override
    public boolean equals(Object other)
    {
        return (other instanceof PluginDetailQuery) ? toString().equals(other.toString()) : false;
    }

    @Override
    public int hashCode()
    {
        return toString().hashCode();
    }
    
    public static class Builder
    {
        private final String pluginKey;
        private Option<Version> version = none();
        private Option<ApplicationKey> application = none();
        private Option<Long> appBuildNumber = none();
        private Option<PricingQuery> includePricing = none();
        private Option<Integer> limitVersions = none();
        private Option<Integer> limitReviews = some(0);
        
        public Builder(String pluginKey)
        {
            this.pluginKey = checkNotNull(pluginKey, "pluginKey");
        }
        
        public PluginDetailQuery build()
        {
            return new PluginDetailQuery(this);
        }
        
        /**
         * Filters the result to only include a specified plugin version or version range.
         * @param version  a {@link Version} object indicating which version(s) to include, or
         *   {@link Option#none} for no version number filtering
         * @return  the same Builder
         */
        public Builder version(Option<Version> version)
        {
            this.version = checkNotNull(version);
            return this;
        }
        
        /**
         * Restricts the result to plugin versions that are compatible with the specified application.
         * This filter is ignored if you used {@link Version#equalTo}) to request a specific version.
         * @param appKey  an {@link ApplicationKey}, or {@link Option#none} for no application filter
         * @return  the same Builder
         */
        public Builder application(Option<ApplicationKey> application)
        {
            this.application = checkNotNull(application);
            return this;
        }
        
        /**
         * Restricts the result to plugin versions that are compatible with the specified application version.
         * This is ignored if you have not specified {@link #appKey}.
         * @param appBuildNumber  the application build number, or {@link Option#none} for no application
         *   version filter
         * @return  the same Builder
         */
        public Builder appBuildNumber(Option<Long> appBuildNumber)
        {
            this.appBuildNumber = checkNotNull(appBuildNumber);
            return this;
        }

        /**
         * Specifies whether to include pricing information in the results for plugins that are
         * purchasable through the Atlassian Marketplace.  By default, none is included, meaning that
         * {@link com.atlassian.marketplace.cient.model.PluginSummary#getPricing} will always
         * return {@link Option#none()}.
         * @param includePricing  a {@link PricingQuery} specifying which prices to include, or
         *   {@link Option#none}
         * @return  the same Builder
         */
        public Builder includePricing(Option<PricingQuery> includePricing)
        {
            this.includePricing = checkNotNull(includePricing);
            return this;
        }
        
        /**
         * Specifies whether to include details for all versions of the plugin or a subset thereof
         * in the {@link com.atlassian.marketplace.client.model.Plugin#getVersions} property of the
         * returned plugin. The default is to include them all (as long as they match other criteria
         * such as {@link #appBuildNumber}, if any). This will not affect the
         * {@link com.atlassian.marketplace.client.model.Plugin#getVersion} property, which always
         * contains the most recent version that matches your search criteria. 
         * @param limitVersions  the maximum number of versions to return; <tt>some(0)</tt> for no
         *   versions; {@link Option#none} for as many versions as possible (the default)
         * @return  the same Builder
         */
        public Builder limitVersions(Option<Integer> limitVersions)
        {
            this.limitVersions = checkNotNull(limitVersions);
            return this;
        }
        
        /**
         * Specifies whether to include details for all reviews of the plugin or a subset thereof
         * in the {@link com.atlassian.marketplace.client.model.Plugin#getReviews} property of the
         * returned plugin. The default is not to include any. This will not affect the
         * {@link com.atlassian.marketplace.client.model.Plugin#getReviewSummary} property, which
         * is always present. 
         * @param limitReviews  the maximum number of reviews to return; <tt>some(0)</tt> for no
         *   reviews (the default); {@link Option#none} for as many reviews as possible
         * @return  the same Builder
         */
        public Builder limitReviews(Option<Integer> limitReviews)
        {
            this.limitReviews = checkNotNull(limitReviews);
            return this;
        }
    }
}
