package com.atlassian.marketplace.client;

import static com.atlassian.marketplace.client.HttpConfiguration.ProxyConfiguration.AuthMethod.BASIC;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Encapsulates all the properties that control the client's HTTP behavior, except for the
 * base URI of the server.
 */
public class HttpConfiguration
{
    public static final int DEFAULT_CONNECT_TIMEOUT_MILLIS = 15000;
    public static final int DEFAULT_READ_TIMEOUT_MILLIS = 15000;
    public static final int DEFAULT_MAX_CONNECTIONS = 10;
    public static final int DEFAULT_MAX_CACHE_ENTRIES = 100;
    public static final long DEFAULT_MAX_CACHE_OBJECT_SIZE = 60000;
    
    private final int connectTimeoutMillis;
    private final int readTimeoutMillis;
    private final int maxConnections;
    private final int maxCacheEntries;
    private final long maxCacheObjectSize;
    private final String username;
    private final String password;
    private final ProxyConfiguration proxy;
    private final RequestDecorator requestDecorator;
 
    /**
     * Returns a new {@link Builder} for constructing an HttpConfiguration instance.
     */
    public static Builder builder()
    {
        return new Builder();
    }
    
    private HttpConfiguration(Builder builder)
    {
        this.connectTimeoutMillis = builder.connectTimeoutMillis;
        this.readTimeoutMillis = builder.readTimeoutMillis;
        this.maxConnections = builder.maxConnections;
        this.maxCacheEntries = builder.maxCacheEntries;
        this.maxCacheObjectSize = builder.maxCacheObjectSize;
        this.username = builder.username;
        this.password = builder.password;
        this.proxy = builder.proxy;
        this.requestDecorator = builder.requestDecorator;
    }
 
    public int getConnectTimeoutMillis()
    {
        return connectTimeoutMillis;
    }
    
    public int getReadTimeoutMillis()
    {
        return readTimeoutMillis;
    }
    
    public int getMaxConnections()
    {
        return maxConnections;
    }
    
    public int getMaxCacheEntries()
    {
        return maxCacheEntries;
    }
    
    public long getMaxCacheObjectSize()
    {
        return maxCacheObjectSize;
    }
    
    public String getUsername()
    {
        return username;
    }
    
    public String getPassword()
    {
        return password;
    }
    
    public boolean hasCredentials()
    {
        return (username != null);
    }
    
    public ProxyConfiguration getProxy()
    {
        return proxy;
    }
    
    public boolean hasProxy()
    {
        return (proxy != null);
    }
    
    public RequestDecorator getRequestDecorator()
    {
        return requestDecorator;
    }
    
    public static class Builder
    {
        private int connectTimeoutMillis = DEFAULT_CONNECT_TIMEOUT_MILLIS;
        private int readTimeoutMillis = DEFAULT_READ_TIMEOUT_MILLIS;
        private int maxConnections = DEFAULT_MAX_CONNECTIONS;
        private int maxCacheEntries = DEFAULT_MAX_CACHE_ENTRIES;
        private long maxCacheObjectSize = DEFAULT_MAX_CACHE_OBJECT_SIZE;
        private String username;
        private String password;
        private ProxyConfiguration proxy;
        private RequestDecorator requestDecorator;
        
        public HttpConfiguration build()
        {
            return new HttpConfiguration(this);
        }
        
        /**
         * Sets the length of time to wait for a connection before timing out.
         * @param connectTimeoutMillis  a number of milliseconds; null to use
         * {@link HttpConfiguration#DEFAULT_CONNECT_TIMEOUT_MILLIS}
         * @return  the same Builder
         */
        public Builder connectTimeoutMillis(Integer connectTimeoutMillis)
        {
            this.connectTimeoutMillis = (connectTimeoutMillis == null) ? DEFAULT_CONNECT_TIMEOUT_MILLIS :
                connectTimeoutMillis.intValue();
            return this;
        }
        
        /**
         * Sets the length of time to wait for a server response before timing out.
         * @param readTimeoutMillis  a number of milliseconds; null to use
         * {@link HttpConfiguration#DEFAULT_READ_TIMEOUT_MILLIS}
         * @return  the same Builder
         */
        public Builder readTimeoutMillis(Integer readTimeoutMillis)
        {
            this.readTimeoutMillis = (readTimeoutMillis == null) ? DEFAULT_READ_TIMEOUT_MILLIS :
                readTimeoutMillis.intValue();
            return this;
        }
        
        /**
         * Sets the maximum number of simultaneous HTTP connections the client can make to the server.
         * @param maxConnections  the maximum number of simultaneous connections; default is
         * {@link HttpConfiguration#DEFAULT_MAX_CONNECTIONS}
         * @return  the same Builder
         */
        public Builder maxConnections(int maxConnections)
        {
            this.maxConnections = maxConnections;
            return this;
        }
        
        /**
         * Sets the maximum number of HTTP responses that will be stored in the HTTP cache at a time.
         * @param maxCacheEntries  maximum number of cache entries; default is
         * {@link HttpConfiguration#DEFAULT_MAX_CACHE_ENTRIES}
         * @return  the same Builder
         */
        public Builder maxCacheEntries(int maxCacheEntries)
        {
            this.maxCacheEntries = maxCacheEntries;
            return this;
        }
        
        /**
         * Sets the maximum size of HTTP responses that can be stored in the HTTP cache.
         * @param maxCacheObjectSize  maximum cacheable object size in bytes; default is
         * {@link HttpConfiguration#DEFAULT_MAX_CACHE_OBJECT_SIZE}
         * @return  the same Builder
         */
        public Builder maxCacheObjectSize(long maxCacheObjectSize)
        {
            this.maxCacheObjectSize = maxCacheObjectSize;
            return this;
        }
        
        /**
         * Sets a username for basic authentication.  Authentication is not required for standard use of
         * the client; authenticating with a valid marketplace.atlassian.com account will allow you to see
         * unpublished versions of plugins you have permission to edit.
         * @param username  the username
         * @return  the same Builder
         */
        public Builder username(String username)
        {
            this.username = username;
            return this;
        }
        
        /**
         * Sets a password for basic authentication.  Authentication is not required for standard use of
         * the client; authenticating with a valid marketplace.atlassian.com account will allow you to see
         * unpublished versions of plugins you have permission to edit.
         * @param password  the password
         * @return  the same Builder
         */
        public Builder password(String password)
        {
            this.password = password;
            return this;
        }
        
        /**
         * Sets HTTP proxy server parameters.
         * @param proxy  a {@link ProxyConfiguration} object
         * @return  the same Builder
         */
        public Builder proxy(ProxyConfiguration proxy)
        {
            this.proxy = proxy;
            return this;
        }
        
        /**
         * Specifies an object that can provide custom headers for HTTP requests, e.g. to set the User-Agent header.
         * @param requestDecorator  a {@link RequestDecorator} object
         * @return  the same Builder
         */
        public Builder requestDecorator(RequestDecorator requestDecorator)
        {
            this.requestDecorator = requestDecorator;
            return this;
        }
    }
    
    public static class ProxyConfiguration
    {
        public enum AuthMethod
        {
            BASIC,
            DIGEST,
            NTLM
        };
        
        private final String host;
        private final Integer port;
        private final AuthMethod authMethod;
        private final String username;
        private final String password;
        private final String ntlmDomain;
        private final String ntlmWorkstation;

        public static Builder builder(String host)
        {
            return new Builder(host);
        }
        
        private ProxyConfiguration(Builder builder)
        {
            this.host = builder.host;
            this.port = builder.port;
            this.authMethod = builder.authMethod;
            this.username = builder.username;
            this.password = builder.password;
            this.ntlmDomain = builder.ntlmDomain;
            this.ntlmWorkstation = builder.ntlmWorkstation;
        }

        public String getHost()
        {
            return host;
        }

        public Integer getPort()
        {
            return port;
        }

        public boolean hasAuth()
        {
            return (username != null);
        }

        public AuthMethod getAuthMethod()
        {
            return (authMethod == null) ? BASIC : authMethod;
        }

        public String getUsername()
        {
            return username;
        }

        public String getPassword()
        {
            return password;
        }

        public String getNtlmDomain()
        {
            return ntlmDomain;
        }

        public String getNtlmWorkstation()
        {
            return ntlmWorkstation;
        }
        
        public static class Builder
        {
            private String host;
            private Integer port;
            private AuthMethod authMethod;
            private String username;
            private String password;
            private String ntlmDomain;
            private String ntlmWorkstation;
            
            public Builder(String host)
            {
                this.host = checkNotNull(host, "host");
            }
            
            public ProxyConfiguration build()
            {
                return new ProxyConfiguration(this);
            }
            
            public Builder host(String host)
            {
                this.host = host;
                return this;
            }
            
            public Builder port(Integer port)
            {
                this.port = port;
                return this;
            }
            
            public Builder authMethod(AuthMethod authMethod)
            {
                this.authMethod = authMethod;
                return this;
            }
            
            public Builder username(String username)
            {
                this.username = username;
                return this;
            }
            
            public Builder password(String password)
            {
                this.password = password;
                return this;
            }
            
            public Builder ntlmDomain(String ntlmDomain)
            {
                this.ntlmDomain = ntlmDomain;
                return this;
            }
            
            public Builder ntlmWorkstation(String ntlmWorkstation)
            {
                this.ntlmWorkstation = ntlmWorkstation;
                return this;
            }
        }
    }
}
