package com.atlassian.theme.internal.request;

import com.atlassian.annotations.nullability.ReturnValuesAreNonnullByDefault;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.theme.api.Theme;
import com.atlassian.theme.api.ThemeColorMode;
import com.atlassian.theme.api.request.RequestScopeThemeService;
import com.atlassian.theme.internal.DefaultThemes;
import com.atlassian.theme.internal.api.ThemeService;
import com.atlassian.theme.internal.api.request.RequestScopeThemeOverrideService;
import com.atlassian.theme.internal.api.user.PreferredColorMode;
import com.atlassian.theme.internal.api.user.UserThemeService;
import org.slf4j.Logger;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import static java.util.Arrays.stream;
import static java.util.Objects.isNull;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static org.slf4j.LoggerFactory.getLogger;

@ReturnValuesAreNonnullByDefault
public class DefaultRequestScopeThemeService implements RequestScopeThemeService, RequestScopeThemeOverrideService {
    private static final Logger log = getLogger(DefaultRequestScopeThemeService.class);

    private static final String COLOR_MODE_OVERRIDE_ATTRIBUTE_KEY = "atlassian-theme-color-mode-override";
    private static final String DARK_THEME_OVERRIDE_ATTRIBUTE_KEY = "atlassian-theme-dark-override";
    private static final String LIGHT_THEME_OVERRIDE_ATTRIBUTE_KEY = "atlassian-theme-light-override";
    private static final String MATCH_USER_AGENT_OVERRIDE_ATTRIBUTE_KEY = "atlassian-theme-match-ua-override";

    private final UserManager userManager;
    private final ThemeService themeService;
    private final UserThemeService userThemeService;
    private final boolean originalThemeIsDefaultLightModeTheme;
    private final boolean originalThemeEnabled;
    private final boolean shouldMatchUserAgentByDefault;

    @SuppressWarnings("unused")
    public DefaultRequestScopeThemeService(
            @Nonnull UserManager userManager,
            @Nonnull ThemeService themeService,
            @Nonnull UserThemeService userThemeService,
            boolean originalThemeIsDefaultLightModeTheme,
            boolean originalThemeEnabled,
            boolean shouldMatchUserAgentByDefault
    ) {
        if (originalThemeIsDefaultLightModeTheme && !originalThemeEnabled) {
            throw new IllegalArgumentException("Original theme needs to be enabled for it to be the default");
        }

        this.userManager = requireNonNull(userManager, "userManager");
        this.themeService = requireNonNull(themeService, "themeService");
        this.userThemeService = requireNonNull(userThemeService, "userThemeService");
        this.originalThemeIsDefaultLightModeTheme = originalThemeIsDefaultLightModeTheme;
        this.originalThemeEnabled = originalThemeEnabled;
        this.shouldMatchUserAgentByDefault = shouldMatchUserAgentByDefault;
    }

    @Nonnull
    @Override
    public Theme getPreferredDarkTheme(@Nonnull HttpServletRequest httpServletRequest) {
        requireNonNull(httpServletRequest, "httpServletRequest");

        final Object darkThemeOverride = httpServletRequest.getAttribute(DARK_THEME_OVERRIDE_ATTRIBUTE_KEY);
        if (!isNull(darkThemeOverride) && darkThemeOverride instanceof String) {
            final Optional<Theme> matchingTheme =
                    themeService.findMatchingTheme(ThemeColorMode.DARK, (String) darkThemeOverride);
            if (matchingTheme.isPresent()) {
                return matchingTheme.get();
            }
        }

        return Optional.ofNullable(userManager.getRemoteUserKey(httpServletRequest))
                .flatMap(userThemeService::getPreferredDarkTheme)
                .orElse(DefaultThemes.DARK);
    }

    @Nonnull
    @Override
    public Theme getPreferredLightTheme(@Nonnull HttpServletRequest httpServletRequest) {
        requireNonNull(httpServletRequest, "httpServletRequest");

        final Object lightThemeOverride = httpServletRequest.getAttribute(LIGHT_THEME_OVERRIDE_ATTRIBUTE_KEY);
        if (!isNull(lightThemeOverride) && lightThemeOverride instanceof String) {
            final Optional<Theme> matchingTheme =
                    themeService.findMatchingTheme(ThemeColorMode.LIGHT, (String) lightThemeOverride);
            if (matchingTheme.isPresent()) {
                return matchingTheme.get();
            }
        }

        if (!originalThemeEnabled) {
            return DefaultThemes.LIGHT;
        }

        final Theme defaultTheme = originalThemeIsDefaultLightModeTheme ? DefaultThemes.ORIGINAL : DefaultThemes.LIGHT;

        return Optional.ofNullable(userManager.getRemoteUserKey(httpServletRequest))
                .flatMap(userThemeService::getPreferredLightTheme)
                .orElse(defaultTheme);
    }

    @Nonnull
    @Override
    public String getHtmlAttributesForThisRequest(@Nonnull HttpServletRequest httpServletRequest) {
        final Optional<ThemeColorMode> initialColorMode = getInitialColorModeWithFallbackAndOverride(httpServletRequest);
        final Theme lightTheme = getPreferredLightTheme(httpServletRequest);
        final Theme darkTheme = getPreferredDarkTheme(httpServletRequest);

        // Everything below must be XSS safe
        List<String> attributes = new ArrayList<>();
        // Ordering matches Cloud
        attributes.add("data-theme=\"dark:"+darkTheme.getThemeKey()+" light:"+lightTheme.getThemeKey()+'"');
        if (initialColorMode.isPresent()) {
            attributes.add("data-color-mode=\"" + initialColorMode.get() + '"');
        } else {
            attributes.add("data-color-mode-auto");
        }

        return ' '+String.join(" ", attributes)+' '; // [HTML syntax] safety first
    }

    /**
     * We could introduce yet-another-enum to represent the three states, but the whole point of {@link ThemeColorMode}
     * is for serialisation safety
     * @return {@link Optional#empty()} represents matching, otherwise the color mode it should be
     */
    @Nonnull
    private Optional<ThemeColorMode> getInitialColorModeWithFallbackAndOverride(@Nonnull HttpServletRequest httpServletRequest) {
        requireNonNull(httpServletRequest, "httpServletRequest");

        // both overrides need to be handled first (otherwise they're not overrides)
        boolean matchUaOverride = Optional.ofNullable(httpServletRequest.getAttribute(MATCH_USER_AGENT_OVERRIDE_ATTRIBUTE_KEY))
                .filter(matchUserAgentOverrideAttribute -> matchUserAgentOverrideAttribute instanceof Boolean)
                .map(matchUserAgentOverrideAttribute -> (Boolean) matchUserAgentOverrideAttribute)
                .orElse(false);

        if (matchUaOverride) {
            return empty();
        }

        Optional<ThemeColorMode> override = Optional.ofNullable(httpServletRequest.getAttribute(COLOR_MODE_OVERRIDE_ATTRIBUTE_KEY))
                .filter(colorModeAttribute -> colorModeAttribute instanceof ThemeColorMode)
                .map(colorModeAttribute -> (ThemeColorMode) colorModeAttribute);

        if (override.isPresent()) {
            return override;
        }

        // Then we handle preferences
        Optional<PreferredColorMode> preferredColorMode = getPreferredColorMode(httpServletRequest);

        if (preferredColorMode.isPresent()) {
            switch (preferredColorMode.get()) {
                case MATCHING:
                    return empty();
                case LIGHT:
                    return Optional.of(ThemeColorMode.LIGHT);
                case DARK:
                    return Optional.of(ThemeColorMode.DARK);
            }
        }

        // Default without any user-specific preferences depends on the flag
        return shouldMatchUserAgentByDefault ? empty() : Optional.of(ThemeColorMode.LIGHT);
    }

    private boolean getShouldMatchUserAgent(@Nonnull HttpServletRequest httpServletRequest) {
        requireNonNull(httpServletRequest, "httpServletRequest");



        final Optional<PreferredColorMode> preferredColorMode = getPreferredColorMode(httpServletRequest);
        if (preferredColorMode.isPresent() && preferredColorMode.get().equals(PreferredColorMode.MATCHING)) {
            return true;
        }

        return shouldMatchUserAgentByDefault;
    }

    private Optional<PreferredColorMode> getPreferredColorMode(@Nonnull HttpServletRequest httpServletRequest) {
        requireNonNull(httpServletRequest, "httpServletRequest");
        return Optional.ofNullable(userManager.getRemoteUserKey(httpServletRequest))
                .flatMap(userThemeService::getPreferredColorMode);
    }

    @Override
    public void setColorModeOverride(@Nonnull HttpServletRequest httpServletRequest, @Nullable String colorMode) {
        requireNonNull(httpServletRequest, "httpServletRequest");
        if (isBlankOrNull(colorMode)) {
            return;
        }

        Optional<ThemeColorMode> themeColorMode = stream(ThemeColorMode.values())
                .filter(validColorMode -> validColorMode.toString().equals(colorMode))
                .findAny();

        if (!themeColorMode.isPresent()) {
            if (log.isDebugEnabled()) {
                log.debug("Provided color override '{}' is not one of the valid choices {}", colorMode, Arrays.toString(ThemeColorMode.values()));
            }
            return;
        }

        httpServletRequest.setAttribute(COLOR_MODE_OVERRIDE_ATTRIBUTE_KEY, themeColorMode.get());
    }

    @Override
    public void setDarkThemeOverride(@Nonnull HttpServletRequest httpServletRequest, @Nullable String themeKey) {
        requireNonNull(httpServletRequest, "httpServletRequest");
        if (isBlankOrNull(themeKey)) {
            return;
        }

        httpServletRequest.setAttribute(DARK_THEME_OVERRIDE_ATTRIBUTE_KEY, themeKey);
    }

    @Override
    public void setLightThemeOverride(@Nonnull HttpServletRequest httpServletRequest, @Nullable String themeKey) {
        requireNonNull(httpServletRequest, "httpServletRequest");
        if (isBlankOrNull(themeKey)) {
            return;
        }

        httpServletRequest.setAttribute(LIGHT_THEME_OVERRIDE_ATTRIBUTE_KEY, themeKey);
    }

    @Override
    public void setMatchingUserAgentEnabledOverride(@Nonnull HttpServletRequest httpServletRequest, @Nullable Boolean match) {
        requireNonNull(httpServletRequest, "httpServletRequest");
        if (isNull(match)) {
            return;
        }

        httpServletRequest.setAttribute(MATCH_USER_AGENT_OVERRIDE_ATTRIBUTE_KEY, match);
    }

    private static boolean isBlankOrNull(@Nullable String string) {
        return isNull(string) || string.trim().isEmpty();
    }
}
