package com.atlassian.velocity.allowlist.uberspect;

import com.atlassian.event.api.EventListener;
import com.atlassian.plugin.event.events.PluginFrameworkStartedEvent;
import com.atlassian.velocity.allowlist.api.internal.PluginAllowlist;
import org.apache.velocity.runtime.RuntimeServices;
import org.apache.velocity.runtime.log.Log;
import org.apache.velocity.util.introspection.SecureIntrospectorImpl;

import java.lang.reflect.Method;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Extension of {@link SecureIntrospectorImpl} which can additionally consult the {@link PluginAllowlist}. In addition
 * to configuring this Introspector for use in the Velocity engine, the engine internal instance must then be retrieved
 * and initialised by calling {@link #setPluginAllowlist} and registered as an event listener.
 *
 * @since 6.0.0
 */
public class PluginAwareSecureIntrospector extends SecureIntrospectorImpl {

    public static final String ALLOWLIST_DEBUG_PROPERTY = "atlassian.velocity.method.allowlist.debug";
    public static final String ALLOWLIST_DEBUG_PROPERTY_ALT = "atlassian.plugins.velocity.method.allowlist.debug";

    private final boolean allowlistDebugMode = Boolean.getBoolean(ALLOWLIST_DEBUG_PROPERTY) || Boolean.getBoolean(ALLOWLIST_DEBUG_PROPERTY_ALT);

    private PluginAllowlist pluginAllowlist;
    private volatile ClassLoader pluginClassLoader;
    private final AtomicBoolean isPluginFrameworkStarted = new AtomicBoolean();

    public PluginAwareSecureIntrospector(Log log, RuntimeServices runtimeServices) {
        super(log, runtimeServices);
    }

    public void setPluginAllowlist(PluginAllowlist pluginAllowlist) {
        this.pluginAllowlist = pluginAllowlist;
    }

    @Override
    protected Class<?> loadClass(String name) throws ClassNotFoundException {
        try {
            return super.loadClass(name);
        } catch (ClassNotFoundException e) {
            if (pluginClassLoader != null) {
                return pluginClassLoader.loadClass(name);
            }
            throw e;
        }
    }

    @EventListener
    public void onPluginFrameworkStarted(PluginFrameworkStartedEvent event) {
        pluginClassLoader = event.getPluginAccessor().getClassLoader();
        isPluginFrameworkStarted.set(true);
    }

    @Override
    protected boolean isIntrospectorEnabled() {
        return isPluginFrameworkStarted.get();
    }

    @Override
    protected boolean isAllowlistDebugMode() {
        return super.isAllowlistDebugMode() || allowlistDebugMode;
    }

    @Override
    protected boolean isAllowlistedClassPackageCached(Class<?> clazz) {
        return super.isAllowlistedClassPackageCached(clazz) ||
                (pluginAllowlist != null && pluginAllowlist.isAllowlistedClassPackage(clazz));
    }

    @Override
    protected boolean isAllowlistedMethodCached(Method method) {
        return super.isAllowlistedMethodCached(method) ||
                (pluginAllowlist != null && pluginAllowlist.isAllowlistedMethod(method));
    }
}
