define('branchUtils/merge-help', [
    'jquery',
    'lodash',
    'stash/api/util/navbuilder',
    'util/events'
], function (
    $,
    _,
    nav,
    events
    ) {

    /**
     * Replaces the merge conflict help dialog contents with more helpful instructions when branch model can be used
     * to interpret the type of merge conflict (auto-merge conflict, conflict for stable->X PR or conflict for
     * unstable -> X PR)
     *
     * @param {PullRequest} pullRequest the current pull request
     * @param {boolean} conflicted whether there are merge conflicts for the pull request
     * @param {map} properties any properties provided by plugins
     */
    function replaceMergeInstructions(pullRequest, conflicted, properties) {
        if (conflicted && properties) {
            if (properties['branch-utils.auto-merge-conflict']) {
                $('#merge-help-dialog .merge-conflicted').html(
                    stash.feature.automerge.autoMergeConflictInstructions({
                        'sourceBranch': pullRequest.getFromRef().getDisplayId(),
                        'targetBranch': pullRequest.getToRef().getDisplayId()
                    })
                );
            } else {
                var mergeInstructions = null;
                var stability = properties['branch-utils.from-ref-stability'];
                if (stability === 'stable') {
                    mergeInstructions = stash.feature.pullRequest.mergeInstructionsForStableSourceBranch;
                } else if (stability === 'unstable') {
                    mergeInstructions = stash.feature.pullRequest.mergeInstructionsForUnstableSourceBranch;
                }

                if (mergeInstructions) {
                    var sourceRepo = pullRequest.getFromRef().getRepository();
                    var targetRepo = pullRequest.getToRef().getRepository();
                    var sourceRemote = null;
                    var targetRemote = null;

                    if (!sourceRepo.isEqual(targetRepo)) {
                        sourceRemote = nav.project(sourceRepo.getProject()).repo(sourceRepo).clone(sourceRepo.getScmId()).buildAbsolute();
                        targetRemote = nav.project(targetRepo.getProject()).repo(targetRepo).clone(targetRepo.getScmId()).buildAbsolute();
                    }

                    $('#merge-help-dialog .merge-conflicted').html(
                        mergeInstructions({
                            'sourceBranch': pullRequest.getFromRef().getDisplayId(),
                            'targetBranch': pullRequest.getToRef().getDisplayId(),
                            'sourceRemote': sourceRemote,
                            'targetRemote': targetRemote
                        })
                    );
                }
            }
        }
    }

    return {
        init : function() {
            var replaceMergeInstructionsOnce;

            events.on('stash.pull-request.cant.merge', function(pullRequest, conflicted, vetoes, properties) {
                replaceMergeInstructionsOnce = _.once(function() {
                    replaceMergeInstructions(pullRequest, conflicted, properties);
                });
            });
            events.on('stash.pull-request.show.cant.merge.help', function() {
                if (replaceMergeInstructionsOnce) {
                    replaceMergeInstructionsOnce();
                }
            });
        }
    };
});
