package com.atlassian.seraph.cookie;

import org.apache.commons.lang.StringUtils;

import java.text.SimpleDateFormat;
import java.util.Date;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletResponse;

/**
 * Until we get to Servlet Spec 3.0, we have to roll our own HTTP only support.
 */
public class HttpOnlyCookieKit
{
    private static final String COOKIE_EXPIRE_DF = "EEE, d MMM yyyy HH:mm:ss 'GMT'";

    /**
     * See http://tools.ietf.org/html/rfc2965 and http://www.ietf.org/rfc/rfc2109.txt and
     * http://en.wikipedia.org/wiki/HTTP_cookie for more info
     * <p/>
     * I built the code here by reading the spec and then onbserving how Tomcat 5.5 re-acts when you set the versions
     * and so on.
     *
     * @param httpServletResponse the response in play
     * @param cookie              containing the raw values we want
     */
    public static void setCookie(HttpServletResponse httpServletResponse, Cookie cookie)
    {
        boolean greaterThanV0 = cookie.getVersion() > 0;

        StringBuilder sb = new StringBuilder();

        sb.append(cookie.getName()).append("=").append(esc(cookie.getValue()));
        if (greaterThanV0)
        {
            sb.append("; Version=").append(cookie.getVersion());
        }
        if (greaterThanV0 && StringUtils.isNotBlank(cookie.getComment()))
        {
            sb.append("; Comment=").append(esc(cookie.getComment()));
        }
        if (StringUtils.isNotBlank(cookie.getDomain()))
        {
            sb.append("; Domain=").append(esc(cookie.getDomain()));
        }
        if (cookie.getMaxAge() >= 0)
        {
            if (greaterThanV0)
            {
                sb.append("; Max-Age=").append(cookie.getMaxAge());
            }
            else
            {
                sb.append("; Expires=").append(expires(cookie));
            }
        }
        if (StringUtils.isNotBlank(cookie.getPath()))
        {
            sb.append("; Path=").append(esc(cookie.getPath()));
        }
        if (cookie.getSecure())
        {
            sb.append("; Secure");
        }
        sb.append("; HttpOnly");
        httpServletResponse.addHeader("Set-Cookie", sb.toString());
    }

    private static String esc(final String value)
    {
        if (isToken(value))
        {
            return value;
        }
        return "\"" + value + "\"";
    }

    private static String expires(final Cookie cookie)
    {
        final int msMaxAge = cookie.getMaxAge() * 1000;
        SimpleDateFormat df = new SimpleDateFormat(COOKIE_EXPIRE_DF);
        return df.format(new Date(System.currentTimeMillis() + msMaxAge));
    }
    // Note -- disabled for now to allow full Netscape compatibility
    // from RFC 2068, token special case characters
    //
    // private static final String tspecials = "()<>@,;:\\\"/[]?={} \t";

    private static final String tspecials = ",;";

    private static boolean isToken(String value)
    {
        int len = value.length();

        for (int i = 0; i < len; i++)
        {
            char c = value.charAt(i);

            if (c < 0x20 || c >= 0x7f || tspecials.indexOf(c) != -1)
            {
                return false;
            }
        }
        return true;
    }

}
