package com.atlassian.seraph.util;

import java.util.Collection;

/**
 * The PathMapper is used to map file patterns to keys, and find an approriate key for a given file path. The pattern rules are consistent with those
 * defined in the Servlet 2.3 API on the whole. Wildcard patterns are also supported, using any combination of * and ?.
 * <h3>Example</h3>
 * <blockquote><code>
 * PathMapper pm = new PathMapper();<br>
 * <br>
 * pm.put("one","/");<br>
 * pm.put("two","/mydir/*");<br>
 * pm.put("three","*.xml");<br>
 * pm.put("four","/myexactfile.html");<br>
 * pm.put("five","/*\/admin/*.??ml");<br>
 * <br>
 * String result1 = pm.get("/mydir/myfile.xml"); // returns "two";<br>
 * String result2 = pm.get("/mydir/otherdir/admin/myfile.html"); // returns "five";<br>
 * </code></blockquote>
 * <p>
 * Implementation note, this uses the I prefix as the {@link PathMapper} class predates the interface and is already in use in many places. This
 * interface should be preferred to the {@link PathMapper} class though.
 */
public interface IPathMapper
{
    /**
     * Retrieve appropriate key by matching patterns with supplied path.
     */
    String get(String path);

    /**
     * Retrieve all mappings which match a supplied path.
     */
    Collection getAll(String path);

    /**
     * Add a key and appropriate matching pattern.
     */
    void put(final String key, final String pattern);
}