package com.atlassian.seraph.filter;

import com.atlassian.seraph.auth.AuthenticatorException;
import com.atlassian.seraph.interceptor.LoginInterceptor;
import org.apache.log4j.Category;

import java.util.Iterator;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * This is a base filter that logs the user in based on the given username and password.
 * It is designed to be extended to support schemes that pass username and password one way or another.
 * <p/>
 * For further info see superclass.
 */
public abstract class PasswordBasedLoginFilter extends BaseLoginFilter
{
    static final Category log = Category.getInstance(PasswordBasedLoginFilter.class);

    public String login(HttpServletRequest request, HttpServletResponse response)
    {
        final boolean dbg = log.isDebugEnabled();

        String status = LOGIN_NOATTEMPT;

        // check for parameters
        UserPasswordPair userPair = extractUserPasswordPair(request);
        if (userPair == null)
        {
            if (dbg)
            {
                log.debug("UserPasswordPair object was null ???");
            }
            return status;
        }

        // try to login the user if possible
        if (userPair.userName == null || userPair.password == null)
        {
            return status;
        }

        List interceptors = getSecurityConfig().getInterceptors(LoginInterceptor.class);

        if (dbg)
        {
            log.debug("____ Username : '" + userPair.userName + "' and password provided - remember me : " + Boolean.toString(userPair.persistentLogin) + " - attempting login request");
        }
        try
        {
            for (Iterator iterator = interceptors.iterator(); iterator.hasNext();)
            {
                LoginInterceptor loginInterceptor = (LoginInterceptor) iterator.next();
                loginInterceptor.beforeLogin(request, response, userPair.userName, userPair.password, userPair.persistentLogin);
            }

            boolean loggedIn = getAuthenticator().login(request, response, userPair.userName, userPair.password, userPair.persistentLogin);
            status = loggedIn ? LOGIN_SUCCESS : LOGIN_FAILED;
        }
        catch (AuthenticatorException e)
        {
            if (dbg)
            {
                log.debug("An exception occurred authenticating : '" + userPair.userName + "'", e);
            }
            status = LOGIN_FAILED;
        }
        finally
        {
            for (Iterator iterator = interceptors.iterator(); iterator.hasNext();)
            {
                LoginInterceptor loginInterceptor = (LoginInterceptor) iterator.next();
                loginInterceptor.afterLogin(request, response, userPair.userName, userPair.password, userPair.persistentLogin, status);
            }
        }
        return status;
    }

    /**
     * Returns a username password pair for this request. If this request does not contain user credentials - returns null;
     *
     * @param request
     * @return user credentials or null
     */
    abstract UserPasswordPair extractUserPasswordPair(HttpServletRequest request);

    /**
     * represents a username password pair of user credentials.
     */
    public static final class UserPasswordPair
    {
        final String userName;
        final String password;
        final boolean persistentLogin;

        UserPasswordPair(String user, String password, boolean persistentLogin)
        {
            this.userName = user;
            this.password = password;
            this.persistentLogin = persistentLogin;
        }
    }
}
