package com.atlassian.seraph.config;

import com.atlassian.seraph.Initable;
import com.atlassian.seraph.SecurityService;
import com.atlassian.seraph.auth.AuthenticationContext;
import com.atlassian.seraph.auth.AuthenticationContextImpl;
import com.atlassian.seraph.auth.Authenticator;
import com.atlassian.seraph.auth.RoleMapper;
import com.atlassian.seraph.controller.SecurityController;
import com.atlassian.seraph.cookie.CookieFactory;
import com.atlassian.seraph.interceptor.Interceptor;
import com.atlassian.seraph.util.XMLUtils;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.opensymphony.util.ClassLoaderUtil;

import edu.emory.mathcs.backport.java.util.Collections;

import java.io.Serializable;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilderFactory;

/**
 * The main class of Seraph's configuration.
 * <p>
 * This class is a Singleton, access it using SecurityConfigFactory.getInstance().
 */
public class SecurityConfigImpl implements Serializable, SecurityConfig
{
    private static final Logger log = Logger.getLogger(SecurityConfigImpl.class);

    public static final String DEFAULT_CONFIG_LOCATION = "seraph-config.xml";

    private static final int YEAR_IN_SECONDS = 365 * 24 * 60 * 60;

    private final Authenticator authenticator;
    private final RoleMapper roleMapper;
    private final SecurityController controller;
    private final List services;
    private final List interceptors = Collections.synchronizedList(new ArrayList());

    private final String loginURL;
    private final String logoutURL;
    private final String originalURLKey;
    private final String cookieEncoding;
    private final String loginCookieKey;
    private final String linkLoginURL;
    private final String authType;

    private boolean insecureCookie;
    // the age of the auto-login cookie - default = 1 year (in seconds)
    private final int autoLoginCookieAge;

    private final LoginUrlStrategy loginUrlStrategy;
    private final String loginCookiePath;

    public SecurityConfigImpl(String configFileLocation) throws ConfigurationException
    {
        if (configFileLocation != null)
        {
            if (SecurityConfigImpl.log.isDebugEnabled())
            {
                SecurityConfigImpl.log.debug("Config file location passed.  Location: " + configFileLocation);
            }
        }
        else
        {
            configFileLocation = "seraph-config.xml";
            if (SecurityConfigImpl.log.isDebugEnabled())
            {
                SecurityConfigImpl.log.debug("Initialising securityConfig using default configFile: " + configFileLocation);
            }
        }

        try
        {
            final DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            final URL fileUrl = ClassLoaderUtil.getResource(configFileLocation, this.getClass());

            if (fileUrl == null)
            {
                throw new IllegalArgumentException("No such XML file: " + configFileLocation);
            }

            // Parse document
            final Document doc = factory.newDocumentBuilder().parse(fileUrl.toString());
            final Element rootEl = doc.getDocumentElement();

            final NodeList nl = rootEl.getElementsByTagName("parameters");
            final Element parametersEl = (Element) nl.item(0);
            final Map globalParams = getInitParameters(parametersEl);

            loginURL = (String) globalParams.get("login.url");
            linkLoginURL = (String) globalParams.get("link.login.url");
            logoutURL = (String) globalParams.get("logout.url");
            cookieEncoding = (String) globalParams.get("cookie.encoding");
            loginCookiePath = (String) globalParams.get("login.cookie.path");
            authType = (String) globalParams.get("authentication.type");
            insecureCookie = "true".equals(globalParams.get("insecure.cookie"));

            if (globalParams.get("original.url.key") != null)
            {
                originalURLKey = (String) globalParams.get("original.url.key");
            }
            else
            {
                originalURLKey = "seraph_originalurl";
            }

            if (globalParams.get("login.cookie.key") != null)
            {
                loginCookieKey = (String) globalParams.get("login.cookie.key");
            }
            else
            {
                loginCookieKey = "seraph.os.cookie";
            }

            if (globalParams.get("autologin.cookie.age") != null)
            {
                autoLoginCookieAge = Integer.parseInt(globalParams.get("autologin.cookie.age").toString());
            }
            else
            {
                autoLoginCookieAge = SecurityConfigImpl.YEAR_IN_SECONDS;
            }

            // be VERY careful about changing order here, THIS reference is passed out while initializing and so we are partially constructed when
            // clients call us

            authenticator = configureAuthenticator(rootEl);
            controller = configureController(rootEl);
            roleMapper = configureRoleMapper(rootEl);
            services = Collections.unmodifiableList(configureServices(rootEl));
            configureInterceptors(rootEl);
            loginUrlStrategy = configureLoginUrlStrategy(rootEl);
            CookieFactory.init(this);
        }
        catch (final Exception e)
        {
            throw new ConfigurationException("Exception configuring from '" + configFileLocation, e);
        }
    }

    private LoginUrlStrategy configureLoginUrlStrategy(final Element rootEl) throws ConfigurationException
    {
        LoginUrlStrategy loginUrlStrategy = (LoginUrlStrategy) configureClass(rootEl, "login-url-strategy", this);

        if (loginUrlStrategy == null)
        {
            loginUrlStrategy = new DefaultLoginUrlStrategy();
        }
        return loginUrlStrategy;
    }

    private Authenticator configureAuthenticator(final Element rootEl) throws ConfigurationException
    {
        Authenticator authenticator = (Authenticator) configureClass(rootEl, "authenticator", this);

        try
        {
            if (authenticator == null)
            {
                authenticator = (Authenticator) ClassLoaderUtil.loadClass(Authenticator.DEFAULT_AUTHENTICATOR, this.getClass()).newInstance();
                authenticator.init(new HashMap(), this);
            }
        }
        catch (final Exception e)
        {
            throw new ConfigurationException("Could not lookup class: " + Authenticator.DEFAULT_AUTHENTICATOR, e);
        }
        return authenticator;
    }

    private SecurityController configureController(final Element rootEl) throws ConfigurationException
    {
        SecurityController controller = (SecurityController) configureClass(rootEl, "controller", this);

        try
        {
            if (controller == null)
            {
                controller = (SecurityController) ClassLoaderUtil.loadClass(SecurityController.NULL_CONTROLLER, this.getClass()).newInstance();
            }
        }
        catch (final Exception e)
        {
            throw new ConfigurationException("Could not lookup class: " + SecurityController.NULL_CONTROLLER, e);
        }
        return controller;
    }

    private RoleMapper configureRoleMapper(final Element rootEl) throws ConfigurationException
    {
        return (RoleMapper) configureClass(rootEl, "rolemapper", this);
    }

    private static Initable configureClass(final Element rootEl, final String tagname, final SecurityConfig owner) throws ConfigurationException
    {
        try
        {
            final NodeList elementList = rootEl.getElementsByTagName(tagname);

            for (int i = 0; i < elementList.getLength(); i++)
            {
                final Element authEl = (Element) elementList.item(i);
                final String clazz = authEl.getAttribute("class");

                final Initable initable = (Initable) ClassLoaderUtil.loadClass(clazz, owner.getClass()).newInstance();

                final Map params = getInitParameters(authEl);

                initable.init(params, owner);
                return initable;
            }
            return null;
        }
        catch (final Exception e)
        {
            throw new ConfigurationException("Could not create: " + tagname, e);
        }
    }

    // only called from the constructor
    private List configureServices(final Element rootEl) throws ConfigurationException
    {
        final NodeList nl = rootEl.getElementsByTagName("services");
        final List result = new ArrayList();

        if ((nl != null) && (nl.getLength() > 0))
        {
            final Element servicesEl = (Element) nl.item(0);
            final NodeList serviceList = servicesEl.getElementsByTagName("service");

            for (int i = 0; i < serviceList.getLength(); i++)
            {
                final Element serviceEl = (Element) serviceList.item(i);
                final String serviceClazz = serviceEl.getAttribute("class");

                if ((serviceClazz == null) || "".equals(serviceClazz))
                {
                    throw new ConfigurationException("Service element with bad class attribute");
                }

                try
                {
                    SecurityConfigImpl.log.debug("Adding seraph service of class: " + serviceClazz);
                    final SecurityService service = (SecurityService) ClassLoaderUtil.loadClass(serviceClazz, this.getClass()).newInstance();

                    final Map serviceParams = getInitParameters(serviceEl);

                    service.init(serviceParams, this);

                    result.add(service);
                }
                catch (final Exception e)
                {
                    throw new ConfigurationException("Could not getRequest service: " + serviceClazz, e);
                }
            }
        }
        return result;
    }

    private void configureInterceptors(final Element rootEl) throws ConfigurationException
    {
        final NodeList nl = rootEl.getElementsByTagName("interceptors");

        if ((nl != null) && (nl.getLength() > 0))
        {
            final Element interceptorsEl = (Element) nl.item(0);
            final NodeList interceptorList = interceptorsEl.getElementsByTagName("interceptor");

            for (int i = 0; i < interceptorList.getLength(); i++)
            {
                final Element interceptorEl = (Element) interceptorList.item(i);
                final String interceptorClazz = interceptorEl.getAttribute("class");

                if ((interceptorClazz == null) || "".equals(interceptorClazz))
                {
                    throw new ConfigurationException("Interceptor element with bad class attribute");
                }

                try
                {
                    SecurityConfigImpl.log.debug("Adding interceptor of class: " + interceptorClazz);
                    final Interceptor interceptor = (Interceptor) ClassLoaderUtil.loadClass(interceptorClazz, this.getClass()).newInstance();

                    final Map interceptorParams = getInitParameters(interceptorEl);

                    interceptor.init(interceptorParams, this);

                    interceptors.add(interceptor);
                }
                catch (final Exception e)
                {
                    throw new ConfigurationException("Could not getRequest service: " + interceptorClazz, e);
                }
            }
        }
    }

    private static Map getInitParameters(final Element el)
    {
        final Map params = new HashMap();
        final NodeList nl = el.getElementsByTagName("init-param");

        for (int i = 0; i < nl.getLength(); i++)
        {
            final Node initParam = nl.item(i);
            final String paramName = XMLUtils.getContainedText(initParam, "param-name");
            final String paramValue = XMLUtils.getContainedText(initParam, "param-value");
            params.put(paramName, paramValue);
        }
        return Collections.unmodifiableMap(params);
    }

    public void destroy()
    {
        for (final Iterator iterator = services.iterator(); iterator.hasNext();)
        {
            final SecurityService securityService = (SecurityService) iterator.next();
            securityService.destroy();
        }

        for (final Iterator iterator = interceptors.iterator(); iterator.hasNext();)
        {
            ((Interceptor) iterator.next()).destroy();
        }
    }

    /**
     * Do not use in production! Only used in tests, will be removed.
     */
    public void addInterceptor(final Interceptor interceptor)
    {
        interceptors.add(interceptor);
    }

    public List getServices()
    {
        return services;
    }

    public String getLoginURL()
    {
        return loginUrlStrategy.getLoginURL(this, loginURL);
    }

    public String getLinkLoginURL()
    {
        return loginUrlStrategy.getLinkLoginURL(this, linkLoginURL);
    }

    public String getLogoutURL()
    {
        return loginUrlStrategy.getLogoutURL(this, logoutURL);
    }

    public String getOriginalURLKey()
    {
        return originalURLKey;
    }

    public Authenticator getAuthenticator()
    {
        return authenticator;
    }

    public AuthenticationContext getAuthenticationContext()
    {
        return new AuthenticationContextImpl(); // todo - load this from a config file
    }

    public SecurityController getController()
    {
        return controller;
    }

    public RoleMapper getRoleMapper()
    {
        return roleMapper;
    }

    public List getInterceptors(final Class desiredInterceptorClass)
    {
        final List result = new ArrayList();

        for (final Iterator iterator = interceptors.iterator(); iterator.hasNext();)
        {
            final Interceptor interceptor = (Interceptor) iterator.next();

            if (desiredInterceptorClass.isAssignableFrom(interceptor.getClass()))
            {
                result.add(interceptor);
            }
        }

        return Collections.unmodifiableList(result);
    }

    public String getCookieEncoding()
    {
        return cookieEncoding;
    }

    public String getLoginCookiePath()
    {
        return loginCookiePath;
    }

    public String getLoginCookieKey()
    {
        return loginCookieKey;
    }

    public String getAuthType()
    {
        return authType;
    }

    public boolean isInsecureCookie()
    {
        return insecureCookie;
    }

    public int getAutoLoginCookieAge()
    {
        return autoLoginCookieAge;
    }
}
