package com.atlassian.seraph.util;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;

/**
 * Caches the results of the {@link PathMapper}
 */
public class CachedPathMapper extends PathMapper
{
    private Map cacheMap;
    private Map cacheAllMap;

    /**
     * Creates a CachedPathMapper object that will use cacheMap to cache the results of the {@link #get(String)} calls
     * and cacheAllMap to cache the results of the {@link #getAll(String)} class.
     * 
     * @param cacheMap
     * @param cacheAllMap
     */
    public CachedPathMapper(Map cacheMap, Map cacheAllMap)
    {
        // Synchronize access to the map for multi-threaded access
        this.cacheMap = Collections.synchronizedMap(cacheMap);
        this.cacheAllMap = Collections.synchronizedMap(cacheAllMap);
    }

    public String get(String path)
    {
        // Check the cache
        String result = (String) cacheMap.get(path);
        if (result != null)
        {
            // The result for this path is cached, return the value
            return result;
        }
        // Get the result from PathMapper
        result = super.get(path);
        // Cache the result
        cacheMap.put(path, result);
        return result;
    }

    public Collection getAll(String path)
    {
        Collection result = (Collection) cacheAllMap.get(path);
        // Check the cache
        if (result != null)
        {
            // The result for this key is cached, return the value
            return result;
        }
        // Get the result from PathMapper
        result = super.getAll(path);
        // Cache the result
        cacheAllMap.put(path, result);
        return result;
    }

    public void put(String key, String pattern)
    {
        cacheMap.remove(key);
        cacheAllMap.remove(key);
        // Let PathMapper update the patterns
        super.put(key, pattern);
    }
}