package com.atlassian.seraph.service;

import com.atlassian.seraph.util.PathMapper;
import com.atlassian.seraph.SecurityService;
import com.atlassian.seraph.config.SecurityConfig;
import com.atlassian.seraph.util.XMLUtils;
import com.atlassian.seraph.util.CachedPathMapper;
import com.opensymphony.util.ClassLoaderUtil;
import org.apache.log4j.Category;
import org.apache.commons.collections.LRUMap;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import javax.servlet.http.HttpServletRequest;
import javax.xml.parsers.DocumentBuilderFactory;
import java.net.URL;
import java.util.*;

/**
 * Configures Seraph to require certain roles to access certain URL paths.
 * <p>
 * Single init-param 'config.file' which is the location of the XML config file.
 * Default value is '/seraph-paths.xml' (loaded from classpath - usually in /WEB-INF/classes)
 * <p>
 * Here's a sample of the XML config file. Path names must be unique
 *<p>
 * <pre>
 * &lt;seraph-paths>
 *	&lt;path name="admin">
 * 		&lt;url-pattern>/secure/admin/*&lt;/url-pattern>
 * 		&lt;role-name>administrators&lt;/role-name>
 * 	&lt;/path>
 * 	&lt;path name="secured">
 *		&lt;url-pattern>/secure/*&lt;/url-pattern>
 *		&lt;role-name>users&lt;/role-name>
 *	&lt;/path>
 * &lt;/seraph-paths>
 * </pre>
 *
 */
public class PathService implements SecurityService
{
    private static final Category log = Category.getInstance(PathService.class);
    static String CONFIG_FILE_PARAM_KEY = "config.file";

    String configFileLocation = "seraph-paths.xml";

    // maps url patterns to path names
    private PathMapper pathMapper = new CachedPathMapper(new LRUMap(10), new LRUMap(10));

    // maps roles to path names
    private Map paths = new HashMap();


    /**
     * Init the service - configure it from the config file
     */
    public void init(Map params, SecurityConfig config)
    {
        try
        {
            if (params.get(CONFIG_FILE_PARAM_KEY) != null)
            {
                configFileLocation = (String) params.get(CONFIG_FILE_PARAM_KEY);
            }

            configurePathMapper();
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }


    /**
     * Configure the path mapper
     */
    private void configurePathMapper()
    {
        try
        {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            URL fileUrl = ClassLoaderUtil.getResource(configFileLocation, this.getClass());

            if (fileUrl == null)
                fileUrl = ClassLoaderUtil.getResource("/" + configFileLocation, this.getClass());


            if (fileUrl == null)
                return;

            // Parse document
            org.w3c.dom.Document doc = factory.newDocumentBuilder().parse(fileUrl.toString());

            // Get list of actions
            Element root = doc.getDocumentElement();
            NodeList pathNodes = root.getElementsByTagName("path");

            // Build list of views
            for (int i = 0; i < pathNodes.getLength(); i++)
            {
                Element path = (Element) pathNodes.item(i);

                String pathName = path.getAttribute("name");
                String roleNames = XMLUtils.getContainedText(path, "role-name");
                String urlPattern = XMLUtils.getContainedText(path, "url-pattern");

                if (roleNames != null && urlPattern != null)
                {
                    String[] rolesArr = parseRoles(roleNames);
                    paths.put(pathName, rolesArr);
                    pathMapper.put(pathName, urlPattern);
                }
            }
        }
        catch (Exception ex)
        {
            log.error(ex);
        }
    }

    protected String[] parseRoles(String roleNames) {
        StringTokenizer st = new StringTokenizer(roleNames, ",; \t\n", false);
        String[] roles = new String[st.countTokens()];
        int i = 0;
        while (st.hasMoreTokens()) {
            roles[i] = st.nextToken();
            i++;
        }
        return roles;
    }

    public void destroy()
    {
    }

    public Set getRequiredRoles(HttpServletRequest request)
    {
        String servletPath = request.getServletPath();
        return getRequiredRoles(servletPath);
    }

    public Set getRequiredRoles(String servletPath)
    {
        Set requiredRoles = new HashSet();

        // then check path mappings first and add any required roles
        Collection constraintMatches = pathMapper.getAll(servletPath);

        if (constraintMatches == null) throw new RuntimeException("No constraints matched for path "+servletPath);
        for (Iterator iterator = constraintMatches.iterator(); iterator.hasNext();)
        {
            String constraint = (String) iterator.next();

            String[] rolesForPath = (String[]) paths.get(constraint);
            for (int i = 0; i < rolesForPath.length; i++)
            {
                if (!requiredRoles.contains(rolesForPath[i])) // since requiredRoles is a set, isn't this useless?
                {
                    requiredRoles.add(rolesForPath[i]);
                }
            }
        }

        return requiredRoles;
    }
}
