package com.atlassian.seraph.cookie;

import com.atlassian.seraph.config.SecurityConfigFactory;
import com.atlassian.seraph.util.CookieUtils;
import com.atlassian.seraph.util.EncryptionUtils;
import org.apache.log4j.Logger;

import java.net.URLEncoder;
import java.net.URLDecoder;
import java.io.UnsupportedEncodingException;

/**
 * Encrypts the username and password using password-based-encryption (MD5 and DES).
 */
public class EncryptedCookieEncoder implements CookieEncoder
{
    private static final Logger LOG = Logger.getLogger(EncryptedCookieEncoder.class);

    private static final String URL_ENCODING = "UTF-8";

    private static final String SEPARATOR = "^^^";
    
    private final EncryptionUtils encryptionUtils;

    public EncryptedCookieEncoder()
    {
        this(SecurityConfigFactory.getInstance().getCookieEncoding());
    }

    protected EncryptedCookieEncoder(String password)
    {
        if (password == null || password.length() == 0)
        {
            throw new IllegalArgumentException("The password must be specified");
        }
        encryptionUtils = new EncryptionUtils();
        encryptionUtils.setPassword(password);
    }

    public String encodePasswordCookie(String username, String password, String encoding)
    {
        StringBuffer sb = new StringBuffer();
        sb.append(username).append(SEPARATOR).append(password);
        return escapeInvalidCookieCharacters(encryptionUtils.encrypt(sb.toString()));
    }

    public String[] decodePasswordCookie(String cookieVal, String encoding)
    {
        final String[] result = new String[2];
        try
        {
            final String text = encryptionUtils.decrypt(unescapeInvalidCookieCharacters(cookieVal));
            int pos = text.indexOf(SEPARATOR);
            if (pos < 0)
            {
                LOG.info("Successfully decrypted password cookie, but decrypted value '" + text + "' is invalid because separator ('" +
                    SEPARATOR + "') was not found. Returning null.");
                return null;
            }
            result[0] = text.substring(0, pos);
            result[1] = text.substring(pos + SEPARATOR.length());
        }
        catch (RuntimeException ex)
        {
            LOG.info("Password cookie could not be decrypted, trying old insecure method of decoding it");
            return CookieUtils.decodePasswordCookie(cookieVal, encoding);
        }
        return result;
    }

    /**
     * Escape invalid cookie characters, see SER-117
     *
     * @param s the String to escape characters for.
     * @return the encoded string.
     * @see #unescapeInvalidCookieCharacters(String)
     */
    private static String escapeInvalidCookieCharacters(final String s)
    {
        try
        {
            return URLEncoder.encode(s, URL_ENCODING);
        }
        catch (UnsupportedEncodingException e)
        {
            LOG.fatal("UTF-8 encoding unsupported !!?!! How is that possible?", e);
            throw new RuntimeException(e);
        }
    }

    /**
     * Un-escape invalid cookie characters, see SER-117
     *
     * @param s the String to escape characters for.
     * @return the encoded string.
     * @see #escapeInvalidCookieCharacters(String)
     */
    private static String unescapeInvalidCookieCharacters(final String s)
    {
        try
        {
            return URLDecoder.decode(s, URL_ENCODING);
        }
        catch (UnsupportedEncodingException e)
        {
            LOG.fatal("UTF-8 encoding unsupported !!?!! How is that possible?", e);
            throw new RuntimeException(e);
        }
    }
}
